# coding: utf-8
import os
import shutil
import multiprocessing as mp

import sandbox.sandboxsdk.process as sdk_process
from sandbox.projects import resource_types
from sandbox.projects.common import environments
from sandbox.projects.common.juggler import base
# from sandbox.projects.common.nanny import nanny
# from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.misc import DnsType
from sandbox.common.types.client import Tag


class GitTagParameter(base.GitTagParameter):
    default_value = 'stable-1.10'


class BuildSentryNginxBinary(base.BaseBuildJugglerTask):
    """
    Build Nginx statically from sources.
    """

    type = "BUILD_SENTRY_NGINX"
    dns = DnsType.DNS64
    client_tags = Tag.LINUX_PRECISE

    REPO_NAME = "nginx"
    GIT_URL_TEMPLATE = "https://github.yandex-team.ru/InfraComponents/{name}.git"

    PCRE_RESOURCE_ID = 109325093
    LIBOPENSSL_RESOURCE_ID = 195130964

    environment = base.BaseBuildJugglerTask.environment + [
        environments.SandboxLibOpensslEnvironment(),
    ]
    input_parameters = [GitTagParameter]

    def _get_version(self):
        with open(os.path.join(self._checkout_path, 'src', 'core', 'nginx.h')) as header:
            for s in header.read().split('\n'):
                if len(s.split()) > 1 and s.split()[1] == "NGINX_VERSION":
                    return s.split()[-1].strip('"')

    def _get_sandbox(self, name, resource_id, srcpath):
        with self.current_action("Downloading {0} from sandbox".format(name)):
            resource_path = self.sync_resource(resource_id)
        with self.current_action("Extracting sources to '{0}'".format(srcpath)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sdk_process.run_process(
                ["tar", "xf", resource_path, "--strip-components=1", "--directory", srcpath],
                log_prefix="extract_{0}".format(name)
            )

    def _get_git(self, name, url, srcpath):
        with self.current_action("Cloning git repo for {0}".format(name)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sdk_process.run_process(
                ["git", "clone", url, srcpath],
                log_prefix="clone_{0}".format(name)
            )

    def on_execute(self):
        self._checkout()

        pcre_path = self.abs_path("pcre")
        self._get_sandbox("pcre", self.PCRE_RESOURCE_ID, pcre_path)

        openssl_path = self.abs_path("openssl")
        self._get_sandbox("pcre", self.LIBOPENSSL_RESOURCE_ID, openssl_path)

        with self._compiler_environ():
            with self.current_action("Configuring nginx"):
                env = os.environ.copy()
                sdk_process.run_process(
                    [
                        "auto/configure",
                        "--prefix=.",
                        "--with-cc-opt=-O2",
                        "--with-debug",
                        "--with-http_stub_status_module",
                        "--with-http_ssl_module",
                        "--with-ipv6",
                        "--with-threads",
                        "--with-file-aio",
                        "--with-stream",
                        "--with-openssl={}".format(openssl_path),
                        "--with-pcre={}".format(pcre_path),
                    ],
                    work_dir=self._checkout_path, environment=env, log_prefix="configure_nginx"
                )

            with self.current_action("Building nginx"):
                sdk_process.run_process(
                    ["make", "-j", str(mp.cpu_count()), "-C", self._checkout_path],
                    log_prefix="make_nginx", work_dir=self._checkout_path
                )

        version = self._get_version()
        self._make_resource(
            path=os.path.join(self._checkout_path, "objs", "nginx"),
            resource_type=resource_types.NGINX_BINARY,
            description='nginx binary {0}, {1}'.format(self.ctx[GitTagParameter.name], version),
            version=version)

#    def on_release(self, additional_parameters):
#        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
#
#        services = ()
#        if additional_parameters['release_status'] == ReleaseStatus.PRESTABLE:
#            services += ("prestable_juggler_dashboard",)
#            services += tuple(
#                'prestable_juggler_main_{0}'.format(dc.lower())
#                for dc in base.DC_LIST
#            )
#
#        elif additional_parameters['release_status'] == ReleaseStatus.STABLE:
#            services += ("all_juggler_batcher_api", "all_juggler_batcher_frontend", "juggler_dashboard")
#            services += tuple(
#                'production_juggler_main_{0}'.format(dc.lower())
#                for dc in base.DC_LIST
#            )
#
#        self._prepare_nanny_snapshot(
#            service_names=services,
#            services_to_prepare=(),
#            resource_type=resource_types.NGINX_BINARY
#        )


__Task__ = BuildSentryNginxBinary
