# coding: utf-8
import os
import glob
import logging

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common import environments
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process


class GitRefIdParameter(SandboxStringParameter):
    name = 'ref_id'
    description = 'Git ref id'
    default_value = 'master'
    required = True


class GitRefShaParameter(SandboxStringParameter):
    name = 'ref_sha'
    description = 'Git ref SHA'
    default_value = ''
    required = False


class ReleaseParameter(SandboxBoolParameter):
    name = 'release'
    description = 'Create SEPELIB_DIST resource'
    default_value = False


class BuildSepelib(SandboxTask):
    type = 'BUILD_SEPELIB'

    client_tags = ctc.Tag.LINUX_BIONIC

    input_parameters = [
        GitRefIdParameter,
        GitRefShaParameter,
        ReleaseParameter,
    ]

    environment = [
        environments.SandboxJavaJdkEnvironment('1.8.0'),
        environments.SwatZookeeperEnvironment('3.4.6'),
        environments.SwatMongoDbEnvironment('2.4.9'),
    ]

    GIT_URL = 'https://{username}:{password}@bb.yandex-team.ru/scm/nanny/sepelib.git'
    CHECKOUT_PATH = 'sepelib'

    def _checkout(self, oauth_token, dest, ref=None):
        checkout_url = self.GIT_URL.format(username='x-oauth-token', password=oauth_token)
        run_process(
            ['git', 'clone', checkout_url, dest],
            log_prefix='git_clone',
            shell=True,
        )
        if ref is not None:
            run_process(
                ['git', 'checkout', ref],
                work_dir=dest,
                log_prefix='git_checkout',
                shell=True)

    def on_execute(self):
        logging.info('checking out the source code')
        ref_id = self.ctx['ref_id']
        ref_sha = self.ctx.get('ref_sha')
        oauth_token = self.get_vault_data('GBG', 'nanny_robot_bb_oauth_token')
        self._checkout(oauth_token=oauth_token,
                       dest=self.CHECKOUT_PATH,
                       ref=ref_sha or ref_id)

        logging.info('creating virtualenv')
        virtualenv_path = self.path('sepelib_build')
        # create virtualenv using *skynet* python
        run_process(['/skynet/python/bin/virtualenv', virtualenv_path])

        virtualenv_bin_path = os.path.join(virtualenv_path, 'bin')
        pip_path = os.path.join(virtualenv_bin_path, 'pip')
        python_path = os.path.join(virtualenv_bin_path, 'python')
        pip_install_args = [pip_path, 'install', '--index=https://pypi.yandex-team.ru/simple']

        logging.info('upgrading pip')
        run_process('{} install -U ./pip-20.3.4.tar.gz'.format(pip_path), log_prefix='upgrading_pip', work_dir=self.CHECKOUT_PATH, shell=True)

        #pip_install_args.append('--trusted-host pypi.yandex-team.ru')

        logging.info('install setuptools')
        run_process(pip_install_args + ['--upgrade', 'setuptools<42'],
                    log_prefix='install_setuptools',
                    work_dir=self.CHECKOUT_PATH)

        logging.info('collecting sepelib requirements')
        run_process('{} collect-requirements.py > requirements.txt'.format(python_path), log_prefix='collect_requirements', work_dir=self.CHECKOUT_PATH, shell=True)

        logging.info('install requirements.txt')
        run_process(pip_install_args + ['-r', 'requirements.txt'],
                    log_prefix='install_requirements',
                    work_dir=self.CHECKOUT_PATH)

        logging.info('install requirements-dev.txt')
        run_process(pip_install_args + ['-r', 'requirements-dev.txt'],
                    log_prefix='install_requirements_dev',
                    work_dir=self.CHECKOUT_PATH)

        logging.info('run tests')
        env = dict(os.environ)
        path_parts = [virtualenv_bin_path]
        if 'PATH' in env:
            path_parts.append(env['PATH'])
        env_path = ':'.join(path_parts)
        env.update({'PATH': env_path})
        run_process(['./test.sh',
                     '--zookeeper={}'.format(os.environ['ZOOKEEPER_DIR']),
                     ],
                    environment=env,
                    log_prefix='test_sh',
                    work_dir=self.CHECKOUT_PATH)

        if self.ctx.get('release'):
            dist_path = self.path('dist')
            setup_py_names = glob.glob1(self.CHECKOUT_PATH, 'setup-*')
            for setup_py in setup_py_names:
                run_process([python_path, setup_py, 'sdist', '--formats=gztar', '--dist-dir', dist_path],
                            log_prefix=setup_py_names,
                            work_dir=self.CHECKOUT_PATH)

            self.create_resource('sepelib', dist_path, resource_types.SEPELIB_DIST)

            run_process([pip_path, 'install', '--index=http://pypi.yandex-team.ru/simple', 'releaselib'],
                        log_prefix='install_releaselib')
            releaselib_pypi_upload = os.path.join(virtualenv_bin_path, 'releaselib-pypi-upload')
            run_process([releaselib_pypi_upload,
                            'c9170e852eb7449caa307453efc3cea4',
                            'ef0562a3a94e407bb045e5a417daa8f9'] + glob.glob(os.path.join(dist_path, '*')),
                        environment=env,
                        log_prefix='upload_to_pypi')


__Task__ = BuildSepelib
