import json
import logging
import time
import random

from httplib import HTTPConnection

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.projects import resource_types

GENCFG_URL = 'api.gencfg.yandex-team.ru'


def get_group_instances(group):
    conn = HTTPConnection(GENCFG_URL)
    conn.connect()
    conn.request('GET', '/trunk/searcherlookup/groups/' + group + '/instances')
    return json.load(conn.getresponse())['instances']


def gen_maps(instances, timestamp):
    shards = {}
    replicas = {}

    for i in instances:
        shards.setdefault(i['shard_name'])
        replicas.setdefault(i['shard_name'], set())
        replicas[i['shard_name']].add(i['hostname'].replace(i['domain'], '') + ':' + str(i['port']))

        for t in i['tags']:
            if t.startswith('a_tier_'):
                shards[i['shard_name']] = t.replace('a_tier_', '')

    shardmap = ''
    for s, t in shards.iteritems():
        shardmap += '{0} {1} {2}\n'.format(s, s.replace('0000000000', timestamp), t)

    random.seed(537)

    replica_map = {}
    for s, r in replicas.iteritems():
        replica_list = list(r)
        random.shuffle(replica_list)
        replica_map[s.replace('0000000000', timestamp)] = replica_list

    return shardmap, json.dumps(replica_map, indent=4)


class GencfgGroup(SandboxStringParameter):
    name = 'group'
    description = 'GENCFG Group'
    default_value = 'SOME_GROUP'


class Timestamp(SandboxStringParameter):
    name = 'timestamp'
    description = 'Timestamp'
    default_value = str(int(time.time()))


class BuildShardmap(SandboxTask):
    type = 'BUILD_SHARDMAP'
    input_parameters = [
        GencfgGroup,
        Timestamp,
    ]
    client_tags = ctc.Tag.Group.LINUX

    def _make_shardmap(self, data, timestamp, group):
        logging.info('creating shardmap file')

        filename = 'shardmap-{}.map'.format(timestamp)

        with open(filename, 'w') as f:
            f.write(data)

        self.create_resource(
            description='shardmap for group ' + group,
            resource_path=filename,
            resource_type=resource_types.SHARDMAP,
        )

    def _make_replicamap(self, data, group):
        logging.info('creating replicamap file')

        filename = 'replicamap.json'

        with open(filename, 'w') as f:
            f.write(data)

        self.create_resource(
            description='replicamap for group ' + group,
            resource_path=filename,
            resource_type=resource_types.REPLICAMAP,
        )

    def on_execute(self):
        timestamp = self.ctx.get(Timestamp.name)
        group = self.ctx.get(GencfgGroup.name)

        instances = get_group_instances(group)
        shardmap, replicamap = gen_maps(instances, timestamp)

        self._make_shardmap(shardmap, timestamp, group)

        self._make_replicamap(replicamap, group)


__Task__ = BuildShardmap
