# -*- coding: utf-8 -*-

import re
import os

import sandbox.common.types.client as ctc

from sandbox.projects.common.utils import svn_revision

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types


class BuildSpecSearchPerlBundle(SandboxTask):
    type = 'BUILD_SPECSEARCH_PERL_BUNDLE'

    client_tags = ~ctc.Tag.Group.LINUX | ctc.Tag.LINUX_PRECISE

    input_parameters = [SandboxArcadiaUrlParameter]

    def on_execute(self):
        arcadia_url = self.ctx[SandboxArcadiaUrlParameter.name]
        arcadia_src_dir = Arcadia.get_arcadia_src_dir(arcadia_url)
        self.ctx['arcadia_revision'] = svn_revision(arcadia_src_dir)

        work_dir = self.abs_path('workcopy')
        make_folder(work_dir)

        build_cmd = [
            os.path.join(arcadia_src_dir, 'specsearches', 'perl_bundle', 'perl_bundle.sh'),
            arcadia_src_dir, work_dir, 'perl_bundle'
        ]
        run_process(build_cmd, log_prefix='run_perl_bundle')

        result_bundle_file = None
        for item in os.listdir(work_dir):
            file_path = os.path.join(work_dir, item)
            if os.path.isfile(file_path) and re.match('perl_bundle[.].*[.]tar[.]gz', item):
                result_bundle_file = file_path
                break
        if result_bundle_file:
            self.create_resource(
                description=self.descr,
                resource_path=result_bundle_file,
                resource_type=resource_types.SPECSEARCH_PERL_BUNDLE,
                arch=self.arch,
                attributes={'version': self.get_version()}
            )
        else:
            raise SandboxTaskFailureError('Unable to find perl bundle file "perl_bundle.*.tar.gz".')

    def get_version(self):
        (rev, tag, branch) = self.arcadia_info()
        if branch is not None:
            return branch.split('/')[-1].split('-')[-1]
        if tag is not None:
            return tag.split('/')[-2].split('-')[-1]
        else:
            return 'trunk'

    def arcadia_info(self):
        parsed_url = Arcadia.parse_url(self.ctx[SandboxArcadiaUrlParameter.name])
        return parsed_url.revision, parsed_url.tag, parsed_url.branch


__Task__ = BuildSpecSearchPerlBundle
