# coding: utf-8

import filecmp
import os.path
import os
import re
import shutil
import time

import logging
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder, add_write_permissions_for_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.nanny.auto_deploy import AutoNannyDeployTask

from sandbox import sdk2


class WizardDataReportSvnUrlParameter(SandboxStringParameter):
    name = 'wizard_data_svn_url'
    description = 'wizard-data svn url'
    default_value = 'arcadia:/arc/trunk/arcadia/search/wizard/data/fresh'


class CheckWizardDataParameter(SandboxBoolParameter):
    name = 'check_wizard_data'
    description = 'Check wizard data before commit'
    default_value = True


class RestartOnReleaseParameter(SandboxBoolParameter):
    name = 'restart_on_release'
    description = 'Restart nanny service on release'
    default_value = False


class YtOauthVaultName(SandboxStringParameter):
    name = 'yt_oauth_vault_name'
    description = 'YT oauth key vault name'
    default_value = ''


class TvmSecretVaultName(SandboxStringParameter):
    name = 'tvm_secret_vault_name'
    description = 'TVM secret vault name'
    default_value = ''


class BuildSportProxyData(AutoNannyDeployTask, SandboxTask):
    """
        Ищет последний релиз бинарника для сбора данных для sport_proxy (русурс CREATE_SPORT_DATA_EXECUTABLE), запускает его, по
        созданному файлу создает ресурс. Обновляет данные в robots/wizard-data для спортивного колдунщика
    """

    type = 'BUILD_SPORT_PROXY_DATA'

    input_parameters = (WizardDataReportSvnUrlParameter, CheckWizardDataParameter, RestartOnReleaseParameter,
                        YtOauthVaultName, TvmSecretVaultName, )

    def logged_process(self, cmd, shell=False, num_tries=1, env=None):
        cmd_str = cmd if isinstance(cmd, str) else ' '.join(cmd)
        stdout_filename = self.abs_path('stdout.log')
        stdout_log = open(stdout_filename, 'w')
        stderr_filename = self.abs_path('stderr.log')
        stderr_log = open(stderr_filename, 'w')
        process = None
        process_env = os.environ.copy()
        if env:
            process_env.update(env)
        for i in xrange(1, num_tries + 1):
            try:
                process = run_process(cmd, stdout=stdout_log, stderr=stderr_log, wait=True, shell=shell, environment=process_env)
            except Exception as e:
                logging.error('process failed: %s' % e)
                logging.error("It's stderr:\n" + open(stderr_filename).read())
            if (not process or process.returncode) and (i < num_tries):
                time.sleep(3.0)
                logging.info('Retry process: %s' % cmd_str)
            else:
                break
        stdout_log.close()
        stderr_log.close()
        if not process or process.returncode:
            raise SandboxTaskFailureError('Process %s died' % cmd_str)

    def _create_check_wizard_data_task(self):
        wizard_data_dir = self.abs_path('wizard-data')
        patch_path = self.abs_path('sport_proxy.diff')
        diff_files = [
            os.path.join('report', 'gzt', 'src', 'biathlon.data'),
            os.path.join('report', 'gzt', 'ya.make')
        ]
        with open(patch_path, 'w') as patch_log:
            run_process(['svn', 'diff'] + diff_files, work_dir=wizard_data_dir, stdout=patch_log, wait=True)
        if not os.path.isfile(patch_path) or os.path.getsize(patch_path) == 0:
            logging.info('Diff is empty. Skip check')
            return

        patch_resource = self.create_resource('Patch for biathlon.data', patch_path, resource_types.WIZDATA_PATCH)
        self.mark_resource_ready(patch_resource)
        self.ctx['wizard_patch_resource_id'] = patch_resource.id

        revision = Arcadia.info(wizard_data_dir)["commit_revision"]
        logging.info('Building patched runtime with revision ' + str(revision))
        self.create_subtask(
            task_type='WIZARD_RUNTIME_BUILD_PATCHED',
            description='Check wizard runtime with patched biathlon.data',
            input_parameters={
                'arcadia_revision': revision,
                'patch_arcadia': patch_resource.id,
                'begemot_shards': 'Wizard',
            },
            arch='linux',
        )

    def _create_sport_proxy_data(self):
        sport_proxy_dir = self.abs_path('sport_proxy')
        make_folder(sport_proxy_dir)
        add_write_permissions_for_path(sport_proxy_dir)
        os.chdir(sport_proxy_dir)

        resource = sdk2.Resource["CREATE_SPORT_DATA_EXECUTABLE"].find(
            attrs={'released': 'stable'}
        ).first()
        resource_data = sdk2.ResourceData(resource)
        resource_path = str(resource_data.path)
        create_sport_proxy_executable = resource_path

        env = dict()
        if self.ctx["yt_oauth_vault_name"]:
            env['YT_TOKEN'] = sdk2.Vault.data(self.ctx["yt_oauth_vault_name"])
        if self.ctx["tvm_secret_vault_name"]:
            env['SPORT_TVM_SECRET'] = sdk2.Vault.data(self.ctx["tvm_secret_vault_name"])

        version_arg = '--data_version=%d' % self.id
        self.logged_process([create_sport_proxy_executable, version_arg], shell=True, num_tries=3, env=env)

        sport_proxy_data = os.path.join(sport_proxy_dir, "sport_proxy.data")
        result_resource = self.create_resource(self.descr, sport_proxy_data, resource_types.SPORT_WIZARD_DATA)
        self.mark_resource_ready(result_resource)

    def _checkout_wizard_data(self, wizard_data_dir):
        if not os.path.exists(wizard_data_dir):
            make_folder(wizard_data_dir)
        add_write_permissions_for_path(wizard_data_dir)
        os.chdir(wizard_data_dir)

        wizard_data_svn_url_info = Arcadia.info(self.ctx['wizard_data_svn_url'])
        Arcadia.checkout(wizard_data_svn_url_info['url'], wizard_data_dir, depth='immediates')

        report_data_dir = os.path.join(wizard_data_dir, 'report')
        os.chdir(report_data_dir)
        Arcadia.update(report_data_dir, set_depth='infinity')

    def _check_patched_biathlon_data(self, report_data_dir):
        '''
        Perform a simple check, that data types mentioned in rules files were generated in new biathlon.data
        '''
        biathlon_data_file = os.path.join(report_data_dir, 'gzt', 'src', 'biathlon.data')
        rules_dir = os.path.join(report_data_dir, 'rules', 'src')

        needed_data_types = []
        for root, dirs, files in os.walk(rules_dir):
            for fname in filter(lambda f: f[:9] == 'biathlon.', files):
                for ln in open(os.path.join(root, fname)):
                    m = re.search(':(\w+)_data;', ln)
                    if m and m.group(1) not in needed_data_types:
                        needed_data_types.append(m.group(1))
        logging.info('needed_data_types: ' + ', '.join(sorted(needed_data_types)))

        generated_data_types = []
        for ln in open(biathlon_data_file):
            m = re.match('=(\w+):', ln)
            if m and m.group(1) not in generated_data_types:
                generated_data_types.append(m.group(1))
        logging.info('generated_data_types: ' + ', '.join(sorted(generated_data_types)))

        missed_data_types = set(needed_data_types) - set(generated_data_types)
        if missed_data_types:
            raise SandboxTaskFailureError('Some data types not found in biathlon.data: ' + ', '.join(sorted(missed_data_types)))

    def _update_resource_in_ya_make(self, ya_make_path, file_name, new_resource_id):
        ya_make_new_path = ya_make_path + '.new.tmp'
        f = open(ya_make_new_path, 'w')

        resource_replaced = False
        for ln in open(ya_make_path):
            m = re.match('^s*FROM_SANDBOX[^\d]+(\d+).+' + file_name, ln)
            if m:
                old_resource_id = m.group(1)
                ln = ln.replace(old_resource_id, str(new_resource_id))
                resource_replaced = True
            f.write(ln)
        f.close()

        if not resource_replaced:
            raise SandboxTaskFailureError('Resource for file %s not found in ya.make' % file_name)

        os.rename(ya_make_new_path, ya_make_path)

    def _patch_wizard_data(self):
        '''
        Returns True, if wizard data was really patched. If diff was empty, then returns False
        '''
        wizard_data_dir = self.abs_path('wizard-data')
        self._checkout_wizard_data(wizard_data_dir)

        report_data_dir = os.path.join(wizard_data_dir, 'report')
        sport_proxy_dir = self.abs_path('sport_proxy')
        biathlon_data_cur = os.path.join(report_data_dir, 'gzt', 'src', 'biathlon.data')
        biathlon_data_new = os.path.join(sport_proxy_dir, 'biathlon.data')

        if filecmp.cmp(biathlon_data_cur, biathlon_data_new):
            # Files identical, patch will be empty
            return False

        shutil.copy(biathlon_data_new, biathlon_data_cur)
        self._check_patched_biathlon_data(report_data_dir)

        make_all_script = os.path.join(report_data_dir, "bin", "makeall.sh")
        self.logged_process(make_all_script, shell=True)

        # makeall.sh creates new biathlon.gzt, now save it as resource and update ya.make
        shutil.copy(os.path.join(report_data_dir, 'gzt', 'biathlon.gzt'), self.abs_path('biathlon.gzt'))
        biathlon_gzt_resource = self.create_resource('New compiled gazettier for biathlon', 'biathlon.gzt', resource_types.WIZARD_DATA)
        self.mark_resource_ready(biathlon_gzt_resource)
        ya_make_path = os.path.join(report_data_dir, 'gzt', 'ya.make')
        self._update_resource_in_ya_make(ya_make_path, 'biathlon.gzt', biathlon_gzt_resource.id)

        return True

    def _commit_wizard_data(self):
        if 'wizard_patch_resource_id' not in self.ctx:
            logging.info('No patch found. Nothing to commit')
            return

        wizard_data_dir = self.abs_path('wizard-data')
        self._checkout_wizard_data(wizard_data_dir)

        Arcadia.apply_patch_file(wizard_data_dir, self.sync_resource(self.ctx['wizard_patch_resource_id']))

        report_data_dir = os.path.join(self.abs_path('wizard-data'), 'report')
        os.chdir(report_data_dir)
        Arcadia.commit(report_data_dir, 'Auto update report/gzt/src/biathlon.data\nSandbox task id: ' + str(self.id) + '\nSKIP_CHECK', user='zomb-sandbox-rw')

    def on_execute(self):
        with self.memoize_stage.create_sport_proxy_data(commit_on_entrance=False):
            self._create_sport_proxy_data()

        with self.memoize_stage.patch_and_check_wizard_data(commit_on_entrance=False):
            wizard_data_patched = self._patch_wizard_data()
            if wizard_data_patched and self.ctx['check_wizard_data']:
                self._create_check_wizard_data_task()
                utils.wait_all_subtasks_stop()

        with self.memoize_stage.analyze_check_result(commit_on_entrance=False):
            utils.check_subtasks_fails(fail_on_failed_children=True)

        with self.memoize_stage.commit_wizard_data(commit_on_entrance=False):
            self._commit_wizard_data()

    def on_release(self, additional_parameters):
        if not self.ctx['restart_on_release']:
            return

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data(self.owner, 'nanny_robot_oauth_token')
        )

        nanny_client.update_service_sandbox_file(
            service_id='parallel-biathlon-yp',
            task_type=self.type,
            task_id=str(self.id),
            deploy=True
        )


__Task__ = BuildSportProxyData
