# coding: utf-8

import os
import logging
import re
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.process import run_process

from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class ArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    default_value = "arcadia:/arc/trunk/arcadia"


class BuildSportPushApi(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Builds SportPushApi web application:
        - checks out SportPushApi from SVN
        - creates virtualenv with necessary dependencies
    """
    type = 'BUILD_SPORT_PUSH_API'

    input_parameters = [ArcadiaUrlParameter]

    SPORT_PUSH_SVN_PATH = 'yweb/freshness/scripts/sport_push'
    PATH_PACKET = 'sport_push'
    PATH_TGZ = 'sport_push.tar.gz'

    def on_execute(self):
        """
        Plan is:
        * checkout sport_push project from svn
        * create virtualenv
        * install dependencies there
        * create tgz archive
        """
        import virtualenv

        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        revision = Arcadia.get_revision(url_arcadia)
        branch = utils.get_short_branch_name(url_arcadia)
        branch = branch or "UNKNOWN_BRANCH"

        logging.info("ARCADIA {} {}".format(branch, revision))
        assert revision, 'Trying to fetch project from SVN, but no revision specified'

        self.ctx['revision'] = revision
        self.ctx['branch'] = branch

        path_packet = self.path(self.PATH_PACKET)
        path_checkout = os.path.join(path_packet, "sport_push")
        path_virtualenv = os.path.join(path_packet, "sport_push", "env")
        logging.info("Path: {} {} {}".format(path_packet, path_checkout, path_virtualenv))

        self._export_arcadia(self.SPORT_PUSH_SVN_PATH, path_checkout)

        # create virtualenv using *skynet* python
        logging.info('Creating virtualenv')
        virtualenv.create_environment(path_virtualenv, site_packages=True)
        self._install_dependencies(path_virtualenv, path_checkout)
        virtualenv.make_environment_relocatable(path_virtualenv)

        self._make_resource(path_checkout)

    def _install_dependencies(self, path_virtualenv, path_checkout):
        logging.info('install python libs')
        pip_path = os.path.join(path_virtualenv, 'bin', 'pip')
        python_path = os.path.join(path_virtualenv, 'bin', 'python')

        run_process([python_path, pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '-r', os.path.join(path_checkout, 'pip_requirements.txt')],
                     log_prefix='dependencies_install',
                     work_dir=path_virtualenv)

    def _make_resource(self, path_packet):
        logging.info('Create tgz file')
        with tarfile.open(self.path(self.PATH_TGZ), 'w:gz') as tar:
            for entry in os.listdir(path_packet):
                tar.add(os.path.join(path_packet, entry), entry)
        self.create_resource(
            description='SportPushApi tgz r{0}'.format(self.ctx['revision']),
            resource_path=self.path(self.PATH_TGZ),
            resource_type=resource_types.SPORT_PUSH_API,
            arch='linux'
        )

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)


__Task__ = BuildSportPushApi
