# coding: utf-8

import os
import logging
import copy

from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.svn import Svn
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.commands import get_make_name

from sandbox.projects.common import apihelpers

from sandbox.projects.common.nanny import nanny


class ArchBundleParameter(parameters.SandboxBoolParameter):
    """
        Если параметр установлен в True, то
        запускаются чайлд-задачи с тем же контекстом для каждой архитектуры.
    """
    name = 'run_on_each_arch'
    description = 'Run on each arch (aka build_bundle):'


class SvnUrl(parameters.SandboxStringParameter):
    name = 'svn_url'
    description = 'Svn url'
    default_value = 'svn+ssh://svn.yandex.ru/statbox/packages/yandex/statbox-binaries/statbox-push-client'


class BuildStatboxPushClient(nanny.ReleaseToNannyTask, SandboxTask):

    type = 'BUILD_STATBOX_PUSHCLIENT_OLD'

    input_parameters = [ArchBundleParameter, SvnUrl]

    # дополнительные параметры для контекста подзадач при мультиархитектурной сборке
    # ключ - название архитектуры; например, если хотим, чтобы linux-сборка запустилась
    # с дополнительными параметрами, пишем что-то вроде
    # additional_bundle_params = {'linux': {'param1': 1, 'param2': 2}}
    arch_bundle_additional_parameters = {}

    archs_for_bundle = ('freebsd', 'linux', )

    def __export_svn(self):
        Svn.export(self.ctx['svn_url'], '.')

    def bundle_mode(self):
        # чтобы не зациклиться после выхода из WAIT_CHILD
        subtasks = channel.sandbox.list_tasks(parent_id=self.id)
        if subtasks:
            for subtask in subtasks:
                if subtask.is_failure():
                    raise SandboxTaskFailureError('Build in worker {0} failed'.format(subtask.id))

                for resource in apihelpers.list_task_resources(subtask.id):
                    # игнорируем ресурсы логов
                    if resource.type in (resource_types.TASK_LOGS, ):
                        continue

                    # копируем ресурс с машины сабтаска.
                    src_path = self.sync_resource(resource.id)
                    dst_path = os.path.join(resource.arch, resource.file_name)
                    paths.copy_path(src_path, dst_path)
                    copy_resource = self.create_resource(
                        description=resource.description,
                        resource_path=dst_path,
                        resource_type=resource.type,
                        arch=resource.arch
                    )
                    logging.debug('Resource {0} copied to {1}'.format(resource.id, copy_resource))
            return
        subtasks = []
        # запускаем задачи для всех архитектур
        for arch in self.archs_for_bundle:
            # входные параметры точно те же, что и для текущей задачи
            sub_ctx = copy.deepcopy(self.ctx)
            # подзадачи не должны быть build_bundle, иначе возникнет рекурсия
            sub_ctx[ArchBundleParameter.name] = False
            # не делаем уведомления для дочерних задач
            sub_ctx['notify_via'] = ''
            # дополнительные параметры, специфичные для архитектуры
            if self.arch_bundle_additional_parameters and arch in self.arch_bundle_additional_parameters:
                sub_ctx.update(self.arch_bundle_additional_parameters[arch])
            subtask = self.create_subtask(
                task_type=self.type,
                description="Worker for task #{0}:{1}".format(self.id, self.descr),
                input_parameters=sub_ctx,
                arch=arch,
                important=self.important
            )
            subtasks.append(subtask.id)
        self.wait_all_tasks_completed(subtasks)

    def on_execute(self):
        if self.ctx.get(ArchBundleParameter.name):
            return self.bundle_mode()

        # параметр сборки под все архитектуры не задан, просто выполняем задачу
        build_dir = os.path.join(self.abs_path(), 'build_dir')
        paths.make_folder(build_dir)
        os.chdir(build_dir)

        self.create_resource(
            description="Statbox PushClient ",
            resource_path=os.path.join(build_dir, 'bin', 'push-client'),
            resource_type=resource_types.STATBOX_PUSHCLIENT,
        )
        self.__export_svn()

        cmake_params = ''
        if self.host == 'rtyserver-test2' or self.host == 'sandbox066':
            cmake = '/opt/cmake/bin/cmake'
            cmake_params += '-DMY_GCC=gcc -DMY_GPP=g++'
        else:
            cmake = 'cmake'

        run_process([cmake, cmake_params, '-DCMAKE_BUILD_TYPE=RelWithDebInfo'], log_prefix='cmake')

        make_cmd = get_make_name(jobs_by_cores=True)
        run_process(make_cmd, log_prefix='make')

    def arcadia_info(self):
        return None, 'Statbox PushClient', None


__Task__ = BuildStatboxPushClient
