# -*- coding: utf-8 -*-

import logging
import os
import os.path
import platform
import sandbox.common.types.client as ctc
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.constants as consts
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects import resource_types
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import (
    SandboxArcadiaUrlParameter, SandboxBoolParameter, SandboxStringParameter)
from sandbox.projects.common.nanny import nanny


class UntestedCodeParameter(SandboxBoolParameter):
    description = 'Use unstable task code (do not enable this option)'
    name = 'use_untested_task_code'
    default_value = False


class RepositoryId(SandboxStringParameter):
    name = "repository_id"
    description = "Repository for dupload"
    default_value = "search-rcss-precise"


class BuildSuggestDeb(nanny.ReleaseToNannyTask, ArcadiaTask):
    """Build the suggest server in arcadia/quality/trailer/suggest,
    run tests in quality/trailer/suggest/tests.

    If not running on SVN trunk (i.e. if pointed to a branch or tag),
    also publish the executable file and configs as resources."""

    type = 'BUILD_SUGGEST_DEB'
    execution_space = (YaMakeTask.execution_space + 40 * 1024)

    input_parameters = [SandboxArcadiaUrlParameter, RepositoryId, UntestedCodeParameter]
    environment = ArcadiaTask.environment

    client_tags = ctc.Tag.Group.LINUX

    def do_execute(self):
        copy_trunk = False
        if self.ctx.get(UntestedCodeParameter.name):
            copy_trunk = True

        arcadia = self.get_arcadia_src_dir(copy_trunk=copy_trunk)
        suggest = 'quality/trailer/suggest'

        tests_dir = os.path.join(suggest, 'tests')
        os.chdir(os.path.join(arcadia, tests_dir))

        results_dir = self.abs_path('results')
        logging.info('Running tests.')
        sdk.do_build(consts.YMAKE_BUILD_SYSTEM, arcadia,
                     [tests_dir], clear_build=False, build_type='debug', test=True, checkout=True)

        logging.info('Building Debian package.')
        daemon_path = os.path.join(suggest, 'web_fastcgi_daemon')
        cmdline_server_path = os.path.join(suggest, 'tests/cmdline_server')
        os.chdir(os.path.join(arcadia, daemon_path))

        daemon_bin_path = os.path.join(results_dir, daemon_path) + '/'

        sdk.do_build(consts.YMAKE_BUILD_SYSTEM, arcadia,
                     [daemon_path], results_dir=results_dir,
                     clear_build=False, build_type='debug')
        copy_path(daemon_bin_path + 'suggest-web-daemon',
                  daemon_bin_path + 'suggest-web-daemon.debug')
        sdk.do_build(consts.YMAKE_BUILD_SYSTEM, arcadia,
                     [daemon_path], results_dir=results_dir,
                     clear_build=True, build_type='release')
        sdk.do_build(consts.YMAKE_BUILD_SYSTEM, arcadia,
                     [cmdline_server_path], results_dir=results_dir,
                     clear_build=True, build_type='release')

        if 'precise' in self.client_info['platform']:
            ubuntu_release = 'precise'
        else:
            ubuntu_release = 'lucid'

        # Save package version to the task context
        version = os.popen('dpkg-parsechangelog | grep ^Version:').read()
        version = version.partition('Version:')[2].strip()
        self.ctx['dpkg-version'] = version

        # Publish executable files as resources
        self.create_resource(
            description='daemon executable v{version} for {release}'.format(
                version=version, release=ubuntu_release),
            resource_path=os.path.join('results', daemon_path, 'suggest-web-daemon'),
            resource_type=resource_types.SUGGEST_WEBDAEMON,
            attributes={
                "ttl": 360,
                "version": version,
                "platform": platform.platform()
            }
        )
        self.create_resource(
            description='cmdline_server v{version} for {release}'.format(
                version=version, release=ubuntu_release),
            resource_path=os.path.join(
                'results', cmdline_server_path, 'cmdline_server'),
            resource_type=resource_types.SUGGEST_CMDLINE_SERVER,
            attributes={
                "ttl": 360,
                "version": version,
                "platform": platform.platform()
            }
        )

        # Publish archived configs for executable as a resource
        run_process('tar -cvf suggest-web-daemon-configs.tar -C configs .', log_prefix='tar_configs')
        copy_path('suggest-web-daemon-configs.tar', self.abs_path('suggest-web-daemon-configs.tar'))
        self.create_resource(
            description='suggest-web-daemon archived configs v{version} for {release}'.format(
                version=version, release=ubuntu_release),
            resource_path='suggest-web-daemon-configs.tar',
            resource_type=resource_types.SUGGEST_WEBDAEMON_CONFIGS,
            attributes={
                "version": version,
                "platform": platform.platform()
            }
        )


__Task__ = BuildSuggestDeb
