# coding: utf-8

import os
import logging
import re
import tarfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class BuildSwitchAdminka(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Build Switch Adminka.
        Checkouts adminka from SVN and
        creates virtualenv with necessary packets.
    """
    type = 'BUILD_SWITCH_ADMINKA'

    input_parameters = [SandboxArcadiaUrlParameter]

    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_PRECISE

    URL_ADMINKA_SVN = 'switch/web'

    URLS_ARCADIA_PACKAGES = [
        # ('library/path/in/arcadia', 'some_package.py'),
    ]

    PATH_PACKET = 'switch'
    PATH_TGZ = 'switch_adminka.tar.gz'

    def arcadia_info(self):
        return self.ctx.get("revision"), "Switch Adminka {} r{}".format(self.ctx.get("branch"), self.ctx.get("revision")), self.ctx.get("branch")

    def on_execute(self):
        """
        Plan is:
        * export django project
        * create virtualenv
        * install dependencies there
        * create tgz archive
        * create shard
        """

        import virtualenv

        url_arcadia = self.ctx[SandboxArcadiaUrlParameter.name]

        revision = Arcadia.get_revision(url_arcadia)
        branch = utils.get_short_branch_name(url_arcadia)
        branch = branch or "UNKNOWN_BRANCH"

        logging.info("ARCADIA {} {}".format(branch, revision))
        assert revision, 'Trying to fetch project from SVN, but no revision specified'

        self.ctx['revision'] = revision
        self.ctx['branch'] = branch

        path_packet = self.path(self.PATH_PACKET)
        path_checkout = os.path.join(path_packet, "switch")
        path_virtualenv = os.path.join(path_packet, "env")

        os.mkdir(path_packet)
        self._export_arcadia(self.URL_ADMINKA_SVN, path_checkout)

        # create virtualenv using *skynet* python
        logging.info('creating virtualenv')
        virtualenv.create_environment(path_virtualenv, site_packages=True)

        self._install_dependencies(path_virtualenv, path_checkout)

        virtualenv.make_environment_relocatable(path_virtualenv)

        self._make_resource(path_packet)

    def _install_dependencies(self, path_virtualenv, path_checkout):
        logging.info('install python libs')
        pip_path = os.path.join(path_virtualenv, 'bin', 'pip')

        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '-r', os.path.join(path_checkout, 'requirements.txt')],
                     log_prefix='dependencies_install',
                     work_dir=path_virtualenv)

    def _make_resource(self, path_packet):
        logging.info('create tgz file')
        with tarfile.open(self.path(self.PATH_TGZ), 'w:gz') as tar:
            for entry in os.listdir(path_packet):
                tar.add(os.path.join(path_packet, entry), entry)
        self.create_resource(
            description='switch adminka tgz r{0}'.format(self.ctx['revision']),
            resource_path=self.path(self.PATH_TGZ),
            resource_type=resource_types.SWITCH_ADMINKA,
            arch='linux'
        )

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[SandboxArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)


__Task__ = BuildSwitchAdminka
