# -*- coding: utf-8 -*-

import os
import shutil
import tarfile
import requests

import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag

from sandbox.projects import resource_types
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxStringParameter

ARCH_NAME_IN = 'templar.tar.gz'
ARCH_NAME_OUT = 'yandex-templar-%s.tar'
TEMPLAR_DIR = 'templar'
URL_PREFIX = 'https://github.yandex-team.ru/nodereport/templar/archive/%s.tar.gz'


class TemplarTag(SandboxStringParameter):
    name = 'templar_tag'
    description = 'Релизный тэг темплара (например v5.2.2)'
    required = True


class BuildTemplarBundle(SandboxTask):
    """
        Таск сборки бандла templar + node_modules
    """
    type = 'BUILD_TEMPLAR_BUNDLE'
    dns = ctm.DnsType.DNS64

    input_parameters = [TemplarTag]

    execution_space = 2048

    client_tags = Tag.LINUX_PRECISE

    environment = (
        environments.GCCEnvironment(),
        environments.NodeJS('4.2.1')
    )

    def on_execute(self):
        path = os.path.join(self.path(), ARCH_NAME_IN)
        tag = self.ctx.get(TemplarTag.name)
        url = URL_PREFIX % tag
        path_out = ARCH_NAME_OUT % tag

        res = requests.get(url, stream=True)
        assert res.status_code == 200
        with open(path, 'wb') as f:
            res.raw.decode_content = True
            shutil.copyfileobj(res.raw, f)

        tar = tarfile.open(path)
        for member in tar.getmembers():
            member.name = '/'.join(member.name.split('/')[1:])
            tar.extract(member, TEMPLAR_DIR)
        tar.close()

        run_process(
            ['npm', 'install', '--production', '--registry=http://npm.yandex-team.ru'],
            work_dir=os.path.join(self.path(), TEMPLAR_DIR)
        )

        tar = tarfile.open(path_out, 'w')
        tar.add(TEMPLAR_DIR)
        tar.close()

        res = self.create_resource(
            description='Bundle (templar: {}, node_modules)'.format(tag),
            resource_path=path_out,
            resource_type=resource_types.TEMPLAR_NODE_MODULES_BUNDLE,
            arch='linux',
            attributes={'ttl': 'inf', 'tag': tag}
        )

        self.mark_resource_ready(res.id)
