# coding: utf-8
import os
import logging
import tarfile
import codecs

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder

import sandbox.projects.common.build.ArcadiaTask
from sandbox.projects import resource_types
from sandbox.projects.common.build import parameters

import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.environments import VirtualEnvironment
from sandbox.sandboxsdk.channel import channel


CHANGELOG_FMT = u"""-----------------------------------------------------------
rev: {rev}
author: {author}
date: {date}

{msg}

{paths}
"""
RESOURCES = '_resources'


class NotifyOnReleaseParameter(sdk_parameters.SandboxStringParameter):
    name = 'notify_on_release'
    description = 'Notify on Release. (Space separated list of E-mails.)'
    required = False
    default_value = 'robot-admins@yandex-team.ru'


class BuildTemplater(sandbox.projects.common.build.ArcadiaTask.ArcadiaTask):
    '''
    Configuration generator
    And iss_hook_install_wrapper
    '''

    type = 'BUILD_TEMPLATER'

    input_parameters = [
        parameters.ArcadiaUrl,
        NotifyOnReleaseParameter,
    ]

    def on_execute(self):
        ARCADIA_PATH = self.get_arcadia_src_dir()
        ARCADIA_REVISION = Arcadia.info(ARCADIA_PATH)['entry_revision']
        ARCADIA_URL = Arcadia.info(ARCADIA_PATH)['url']
        if Arcadia.parse_url(ARCADIA_URL).trunk:
            ARCADIA_BRANCH = 'trunk'
        else:
            ARCADIA_BRANCH = Arcadia.parse_url(ARCADIA_URL).branch.split('/')[-1]
        logging.info('Arcadia path: {}, at rev {} ({})'.format(ARCADIA_PATH, ARCADIA_REVISION, ARCADIA_BRANCH))

        ORIGINAL_VENV_PATH = make_folder(os.path.join("CHECKOUT_DIR", "TEMPLATER"))
        VENV_PATH = os.path.join(ORIGINAL_VENV_PATH, 'venv')
        RESULT_RESOURCE = "templater.{}.{}.tar.gz".format(ARCADIA_REVISION, ARCADIA_BRANCH)
        self.ctx['notify_resouce_name'] = RESULT_RESOURCE
        self.ctx['notify_version'] = "{}.{}".format(ARCADIA_REVISION, ARCADIA_BRANCH)

        # Prepare python virtual environment
        logging.info('STAGE1: Prepare Virtual Env in {}'.format(VENV_PATH))
        with VirtualEnvironment(working_dir=ORIGINAL_VENV_PATH, do_not_remove=True) as venv:
            venv.make_relocatable()
            venv.pip('Jinja2==2.8')
            venv.pip('dnspython==1.12.0')
            venv.pip('urllib3==1.14')

        # Shared Libraries for python. (from skynet)
        ''' broken at the moment22.
        logging.info('STAGE1.1: Prepare python shared libraries')
        run_process(['ls', '-la', os.path.join("/skynet", "python")])
        run_process([
            'cp', '-r',
            os.path.join("/skynet", "python", "lib", "*"),
            os.path.join(VENV_PATH, "lib")
        ])
        '''

        # Templater + Ygetparam
        logging.info('STAGE2: Prepare Files for TEMPLATER')
        run_process([
            'cp', '-r',
            os.path.join(ARCADIA_PATH, "tools", "templater", "templater.py"),
            os.path.join(VENV_PATH, "lib", "python2.7", "site-packages")
        ])
        os.symlink(
            os.path.join("..", "lib", "python2.7", "site-packages", "templater.py"),
            os.path.join(VENV_PATH, "bin", "templater")
        )
        run_process([
            'cp', '-r',
            os.path.join(ARCADIA_PATH, "tools", "ygetparam", "ygetparam.py"),
            os.path.join(ARCADIA_PATH, "tools", "ygetparam", "ygetparam_modules"),
            os.path.join(ARCADIA_PATH, "tools", "ygetparam", "curly_braces_expander.py"),
            os.path.join(VENV_PATH, "lib", "python2.7", "site-packages")
        ])
        os.symlink(
            os.path.join("..", "lib", "python2.7", "site-packages", "ygetparam.py"),
            os.path.join(VENV_PATH, "bin", "ygetparam")
        )
        # Iss_hook_install wrapper
        logging.info('STAGE3: Prepare Files for HOOK wrapper')
        HOOK_PATH = make_folder(os.path.join("CHECKOUT_DIR", "TEMPLATER_ISS_HOOK_INSTALL"))
        iss_hook_install_path = os.path.join(HOOK_PATH, "iss_hool_install_wrapper.{}.{}".format(ARCADIA_REVISION, ARCADIA_BRANCH))
        run_process(['cp', os.path.join(ARCADIA_PATH, "tools", "templater", "iss_hool_install_wrapper.sh"), iss_hook_install_path])

        # Change Log
        logging.info('STAGE4: Generate ChangeLog')
        write_changelog(VENV_PATH,
                        "/".join([ARCADIA_URL, "tools", "templater"]),
                        ARCADIA_REVISION,
                        limit=50)
        write_changelog(VENV_PATH,
                        "/".join([ARCADIA_URL, "tools", "ygetparam"]),
                        ARCADIA_REVISION,
                        limit=50)
        write_changelog(VENV_PATH,
                        "/".join([ARCADIA_URL, "tools", "templater", "iss_hool_install_wrapper.sh"]),
                        ARCADIA_REVISION,
                        limit=50)

        # Create Resources
        logging.info('STAGE5: Create resouces')
        resources = []
        with tarfile.open(name=RESULT_RESOURCE, mode='w:gz') as tar:
            for entry in os.listdir(VENV_PATH):
                tar.add(os.path.join(VENV_PATH, entry), entry)
        resource = self.create_resource(
            description="Templater script with environment",
            resource_path=RESULT_RESOURCE,
            # self.GetContextValue('resource.type', default="TEAMCITY_RESOURCE")
            resource_type=resource_types.TEMPLATER,
            arch='linux'
        )
        resources.append(resource.id)
        resource = self.create_resource(
            description="Iss_hook_install wrapper script",
            resource_path=iss_hook_install_path,
            resource_type=resource_types.TEMPLATER_ISS_HOOK_INSTALL,
            arch='linux'
        )
        resources.append(resource.id)
        self.ctx[RESOURCES] = {'resources': resources}

    def on_release(self, additional_parameters):
        for resource_id in self.ctx[RESOURCES]['resources']:
            channel.sandbox.set_resource_attribute(resource_id, 'ttl', 'inf')
        # Notify
        if self.ctx['notify_on_release'] and len(self.ctx['notify_on_release']):
            channel.sandbox.send_email(
                self.ctx['notify_on_release'].split(),
                None,
                "New Templater version released. {}".format(self.ctx['notify_version']),
                "Task: {}\nChangelog: https://proxy.sandbox.yandex-team.ru/task/{}/{}/changelog".format(self.http_url(), self.id, self.ctx['notify_resouce_name']))


def write_changelog(changelog_dir, url, rev_to, **kwargs):
    changelog_dir_path = make_folder(os.path.join(changelog_dir, "changelog"))
    changelog_file_name = os.path.join(changelog_dir_path, "{}.changelog".format(url.split("/")[-1]))
    change_log = Arcadia.log(url, 0, rev_to, kwargs)
    with codecs.open(changelog_file_name, "w", encoding='utf-8') as fd:
        for commit in sorted(change_log, key=lambda x: x['revision'], reverse=True):
            files_in_commit = ""
            for path in commit['paths']:
                files_in_commit = "{}  {}  {}\n".format(files_in_commit, path[0], path[1])
            fd.write(
                CHANGELOG_FMT.format(
                    rev=commit['revision'], author=commit['author'],
                    date=commit['date'], msg=commit['msg'], paths=files_in_commit
                )
            )


__Task__ = BuildTemplater
