# coding: utf-8
import os
import re

from sandbox.projects import resource_types
from sandbox.projects.common.BaseBuildPythonBundleTask import BaseBuildPythonBundleTask, Utils

from sandbox.sandboxsdk.parameters import SandboxGitUrlParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class TimelineRepositoryPath(SandboxGitUrlParameter):
    name = 'timeline_repository_path'
    description = 'GIT repository with TimeLine code'
    default_value = 'ssh://git@bb.yandex-team.ru/search_infra/timeline.git'


class BuildTimeline(BaseBuildPythonBundleTask):
    """
        Задача сборки timeline.
    """
    type = 'BUILD_TIMELINE'

    # Deploy service after configuration update on task release.
    auto_deploy = False

    # Project code name is used for code archive and resources names generation.
    CODE_NAME = 'timeline'
    VENV_ARCHIVE_FILE = 'timeline.tar.gz'
    # 'Name' part of shard full name: '<name>-<tag>-<timestamp>'
    VENV_SHARD_NAME = 'timeline'

    RESOURCE_TYPE_VENV_ARCHIVE = resource_types.TIMELINE
    RESOURCE_TYPE_VENV_SHARD = resource_types.TIMELINE_SHARD

    REPO_URL_PARAMETER = TimelineRepositoryPath

    VAULT_OWNER = "RUNTIME"
    SSH_KEY_NAME = "robot-switter-ssh-key"

    def _check_prerequisites(self):
        ref = self.ctx.get(self.REPO_BRANCH_PARAMETER.name)
        if not ref:
            error_msg = "Can't set timeline version: no git reference given (branch, tag or even commit hash).\n" \
                        "\tHere is a list of most recent version tags: {}".format(self.git.list_tags()[-4:0])
            raise SandboxTaskFailureError(error_msg)

    @property
    def checkout_version(self):
        return self.ctx.get(self.REPO_BRANCH_PARAMETER.name)

    def _set_timeline_version(self):
        version_file_path = os.path.join('timeline', '__init__.py')

        pointer = self.checkout_version

        code_version = re.sub(r'/', r'_', pointer)

        cmd = 'sed -i "s/v0.0/{}/" {}'.format(code_version, version_file_path)

        Utils.run_cmd(cmd, log_message='Setting timeline version info: {cmd}', log_prefix='set-timeline-version',
                      work_dir=self.TEMP_CODE_PATH)

    def before_execution(self, common_attributes):
        self._check_prerequisites()

    def venv_after_requirements(self, venv, requirements_path):
        self._set_timeline_version()

        # Install timeline itself
        venv.pip('{}'.format(self.TEMP_CODE_PATH))

    def get_stable_services(self):
        self.update_comment = 'Timeline version: '+str(self.checkout_version)+'. Update {shard} to {task_id} task id.'
        return ['production_timeline']


__Task__ = BuildTimeline
