# coding=utf-8

import logging
import os
from shutil import copytree
from tempfile import mktemp

import sandbox.common.types.misc as ctm
import sandbox.sdk2.path as spath
from sandbox import sdk2
from sandbox.projects.BuildUchenki.resource_types import UchenkiFrontend
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.sandboxsdk import environments
from sandbox.projects.common.vcs import aapi
from sandbox.sdk2.helpers import subprocess as sp


class BuildUchenkiFrontend(sdk2.Task):
    """
    npm run build sandbox style
    """

    __PATH_TO_UI = 'search/mon/uchenki/ui'
    __BUILD_RESULT_DIR = 'out'

    class Parameters(sdk2.Task.Parameters):

        arcadia_revision = sdk2.parameters.String('Arcadia revision', default=None, required=False)
        checkout_arcadia_from_url = sdk2.parameters.String('Svn url for arc with revision', default=sdk2.svn.Arcadia.ARCADIA_TRUNK_URL, required=True)
        arcadia_patch = sdk2.parameters.String('Apply patch (text diff or rbtorrent) or arc:id', multiline=True, default='')
        environment = sdk2.parameters.String('NPM environment', default='production')

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [
            environments.NodeJS('10.20.1'),
            environments.GCCEnvironment('5.3.0'),
        ]

    def _prepare_env(self):
        """
        Calling prepare for environments
        :return:
        """
        for env in self.Requirements.environments:
            env.prepare()

    def build_npm_and_upload(self, target, revision=None):
        logger = logging.getLogger('npm_builder')
        revision = revision or aapi.ArcadiaApi.svn_head()
        logger.info('Building {} revision: {}'.format(target, revision))

        build_path = os.path.join(os.getcwd(), 'build_dir')

        with arcadiasdk.mount_arc_path(sdk2.svn.Arcadia.replace(self.Parameters.checkout_arcadia_from_url, revision=revision)) as a_arcadia_path:
            if getattr(self.Parameters, 'arcadia_patch', ''):
                # patch = self.Parameters.arcadia_patch
                # if patch.startswith("arc:"):
                #     patch = sdk2.svn.Arcadia.apply_patch(aarcadia, patch, self.path())
                arcadiasdk.apply_patch(self, a_arcadia_path, self.Parameters.arcadia_patch, self.path())
            copytree(os.path.join(a_arcadia_path, target), build_path)

        frontend_archive_path = mktemp(suffix='.tar.gz', prefix='frontend')

        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            sp.check_call(
                ['npm', '--version'],
                cwd=build_path,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            sp.check_call(
                ['npm', 'install'],
                cwd=build_path,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            sp.check_call(
                ['npm', 'run', 'build:{}'.format(self.Parameters.environment)],
                cwd=build_path,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            sp.check_call(
                ['tar', 'czvf', frontend_archive_path, '.'],
                cwd=os.path.join(build_path, 'out'),
                stdout=pl.stdout,
                stderr=pl.stderr
            )

        resource_type = UchenkiFrontend
        app_res = sdk2.ResourceData(resource_type(
            self, 'Frontend archive', 'frontend.tar.gz'
        ))
        if app_res.path.exists():
            app_res.path.unlink()
        app_res.path.write_bytes(spath.Path(frontend_archive_path).read_bytes())
        app_res.ready()

    def on_execute(self):
        self._prepare_env()
        self.build_npm_and_upload(self.__PATH_TO_UI, revision=self.Parameters.arcadia_revision)
