# -*- coding: utf-8 -*-
import __builtin__

import os
import re

import sandbox.common.types.client as ctc

import resource_types as ukrop_resource_types

from sandbox.projects import resource_types
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build.BaseSimpleBuildTaskWithPkg import BaseSimpleBuildTaskWithPkg
from sandbox.projects.common.arcadia import sdk

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.svn import Arcadia


def target2field_name(target):
    return 'build_%s' % target


class BuildUkrop(BaseSimpleBuildTaskWithPkg):
    type = 'BUILD_UKROP'

    client_tags = ctc.Tag.GENERIC | ctc.Tag.UKROP

    execution_space = 24 << 10  # MB

    # Keep this list sorted by basename of second field.
    TARGET_RESOURCES = (
        (resource_types.UKROP_ACTUALIZE_KIWI_EXPORT, 'yweb/robot/gefest/actualize_kiwi_export/actualize_kiwi_export'),
        (resource_types.UKROP_ARRANGE_CRAWLATTEMPT_LOGS, 'yweb/robot/ukrop/tools/arrange_crawlattempt_logs/arrange_crawlattempt_logs'),
        (resource_types.UKROP_ARRANGE_SAMPLES, 'yweb/robot/ukrop/tools/arrange_samples/arrange_samples'),
        (resource_types.UKROP_BACKUPDUMPER, 'yweb/robot/ukrop/tools/backupdumper/backupdumper'),
        (resource_types.UKROP_CALCQUALITY, 'yweb/robot/ukrop/hostslices/calcquality/calcquality'),
        (ukrop_resource_types.UKROP_PREPARE_REGEX_TEST_DATA, 'yweb/robot/ukrop/hostslices/prepare_regex_test_data/prepare_regex_test_data'),
        (resource_types.UKROP_COMPILESLICES_MR, 'yweb/robot/ukrop/hostslices/compileslices_mr/compileslices_mr'),
        (resource_types.CRAWL_CONFIGS_GENERATOR, 'yweb/robot/tools/crawl_configs_generator/crawl_configs_generator'),
        (resource_types.UKROP_CRUTCH_MRAGENT, 'yweb/robot/ukrop/tools/crutch_mragent/crutch_mragent'),
        (resource_types.UKROP_FACTORS_HANDLER, 'yweb/robot/ukrop/tools/factors_handler/factors_handler'),
        (resource_types.UKROP_FIND_FRESH, 'yweb/robot/ukrop/tools/find_fresh/find_fresh'),
        (resource_types.UKROP_FRESH, 'yweb/robot/ukrop/fresh/server/fresh'),
        (resource_types.UKROP_FRESHCLIENT, 'yweb/robot/ukrop/tools/freshclient/freshclient'),
        (resource_types.UKROP_GEMINI_LOOKUP, 'yweb/robot/ukrop/tools/gemini_lookup/gemini_lookup'),
        (resource_types.UKROP_GETBORDERS, 'yweb/robot/ukrop/tools/getborders/getborders'),
        (resource_types.UKROP_HOSTLIST, 'yweb/robot/ukrop/tools/hostlist/hostlist'),
        (resource_types.KIWI_FORMAT, 'yweb/robot/tools/kiwi_format/kiwi_format'),
        (resource_types.UKROP_MK_TORRENT_DEL_LIST, 'yweb/robot/ukrop/tools/mk_torrent_del_list/mk_torrent_del_list'),
        (resource_types.UKROP_PRINTHISTORECS, 'yweb/robot/ukrop/tools/printhistorecs/printhistorecs'),
        (resource_types.UKROP_PROTOBIN2MR, 'tools/protobin2mr/protobin2mr'),
        (resource_types.UKROP_RTCUSTOMFILTER, 'yweb/robot/ukrop/tools/rtcustomfilter_format/rtcustomfilter_format'),
        (resource_types.UKROP_SPLIT_DATA, 'yweb/robot/mr_bins/split_data/split_data'),
        (resource_types.UKROP_SPLITBYURL, 'yweb/robot/ukrop/tools/splitbyurl/splitbyurl'),
        (resource_types.UKROP_BUILD_TRIE, 'tools/triecompiler/triecompiler'),
        (resource_types.UKROP_MR_INDEX, 'yweb/robot/ukrop/tools/ukrop_mr_index/ukrop_mr_index'),
        (resource_types.UKROP_UNPACK_HOSTDATA, 'yweb/robot/ukrop/tools/unpack_hostdata/unpack_hostdata'),
        (resource_types.UKROP_UNPACK_SAMPLES, 'yweb/robot/ukrop/tools/unpack_samples/unpack_samples'),
        (resource_types.UKROP_ZONEADDER, 'yweb/robot/ukrop/tools/zoneadder/zoneadder'),
        (resource_types.UKROP_MR_FETCHED_COUNTER, 'yweb/robot/ukrop/tools/mr_fetched_counter/mr_fetched_counter'),
        (resource_types.UKROP_MR_TOP_HOST_CRAWL_MON, 'yweb/robot/ukrop/tools/top_host_crawl_mon/top_host_crawl_mon'),
    )

    PKG_CTX = {
        'repo_key_id': 'Ukrop Project (Yandex Crawler) <robot-ukrop-sandbox@yandex-team.ru>',
        'gpg_private_key': 'robot-ukrop-sandbox-gpg-private',
        'gpg_public_key': 'robot-ukrop-sandbox-gpg-public',
        'ssh_private_key': 'robot-ukrop-sandbox-ssh-private',
        'sub_arch_hint': {
            'linux': 'linux_ubuntu_12.04_precise',
        }
    }

    RESOURCE_TO_PKG_MAP = {
        'linux': {
            'CRAWL_CONFIGS_GENERATOR': 'packages/debian/ukrop/yandex-search-ukrop-crawl-configs-generator',
            'KIWI_FORMAT': 'packages/debian/ukrop/yandex-search-ukrop-kiwi-format',
            'UKROP_ACTUALIZE_KIWI_EXPORT': 'packages/debian/ukrop/yandex-search-ukrop-actualize-kiwi-export',
            'UKROP_ARRANGE_CRAWLATTEMPT_LOGS': 'packages/debian/ukrop/yandex-search-ukrop-arrange-crawlattempt-logs',
            'UKROP_ARRANGE_SAMPLES': 'packages/debian/ukrop/yandex-search-ukrop-arrange-samples',
            'UKROP_BACKUPDUMPER': 'packages/debian/ukrop/yandex-search-ukrop-backupdumper',
            'UKROP_BUILD_TRIE': 'packages/debian/ukrop/yandex-search-ukrop-build-trie',
            'UKROP_CALCQUALITY': 'packages/debian/ukrop/yandex-search-ukrop-calcquality',
            'UKROP_PREPARE_REGEX_TEST_DATA': 'packages/debian/ukrop/yandex-search-ukrop-prepare-regex-test-data',
            'UKROP_COMPILESLICES_MR': 'packages/debian/ukrop/yandex-search-ukrop-compileslices-mr',
            'UKROP_CRUTCH_MRAGENT': 'packages/debian/ukrop/yandex-search-ukrop-crutch-mragent',
            'UKROP_FACTORS_HANDLER': 'packages/debian/ukrop/yandex-search-ukrop-factors-handler',
            'UKROP_FIND_FRESH': 'packages/debian/ukrop/yandex-search-ukrop-fresh-bin',
            'UKROP_FRESH': 'packages/debian/ukrop/yandex-search-ukrop-fresh-bin',
            'UKROP_FRESHCLIENT': 'packages/debian/ukrop/yandex-search-ukrop-freshclient',
            'UKROP_GEMINI_LOOKUP': 'packages/debian/ukrop/yandex-search-ukrop-gemini-lookup',
            'UKROP_GETBORDERS': 'packages/debian/ukrop/yandex-search-ukrop-getborders',
            'UKROP_HOSTLIST': 'packages/debian/ukrop/yandex-search-ukrop-hostlist',
            'UKROP_MK_TORRENT_DEL_LIST': 'packages/debian/ukrop/yandex-search-ukrop-mk-torrent-del-list',
            'UKROP_MR_INDEX': 'packages/debian/ukrop/yandex-search-ukrop-mr-index',
            'UKROP_PRINTHISTORECS': 'packages/debian/ukrop/yandex-search-ukrop-printhistorecs',
            'UKROP_PROTOBIN2MR': 'packages/debian/ukrop/yandex-search-ukrop-protobin2mr',
            'UKROP_RTCUSTOMFILTER': 'packages/debian/ukrop/yandex-search-ukrop-rtcustomfilter-format',
            'UKROP_SPLIT_DATA': 'packages/debian/ukrop/yandex-search-ukrop-split-data',
            'UKROP_SPLITBYURL': 'packages/debian/ukrop/yandex-search-ukrop-splitbyurl',
            'UKROP_UNPACK_HOSTDATA': 'packages/debian/ukrop/yandex-search-ukrop-unpack-hostdata',
            'UKROP_UNPACK_SAMPLES': 'packages/debian/ukrop/yandex-search-ukrop-unpack-samples',
            'UKROP_MR_FETCHED_COUNTER': 'packages/debian/ukrop/yandex-search-ukrop-mr-fetched-counter',
            'UKROP_MR_TOP_HOST_CRAWL_MON': 'packages/debian/ukrop/yandex-search-ukrop-tophostcrawlmon',
            # XXX Nobody uses yandex-search-ukrop-zoneadder package now.
            # 'UKROP_ZONEADDER': 'packages/debian/ukrop/yandex-search-ukrop-zoneadder',
        },
    }

    REL_STAT_TO_Z2_CFG_MAP = {
        'unstable': 'UKROP_PRE_STG',
        'testing': 'UKROP_STG',
        'prestable': 'UKROP_PRESTABLE',
    }

    class UkropBinaries(parameters.SandboxInfoParameter):
        name = 'info_field'
        description = 'Ukrop binaries'

    input_parameters = BaseSimpleBuildTaskWithPkg.input_parameters + [UkropBinaries]
    for _, path in TARGET_RESOURCES:
        target = os.path.basename(path)
        field_name = target2field_name(target)
        input_parameters.extend([__builtin__.type(
            field_name,
            (parameters.SandboxBoolParameter,),
            dict(name=field_name, description=target, default_value=True),
        )])

    def initCtx(self):
        BaseSimpleBuildTaskWithPkg.initCtx(self)
        self.ctx['build_system'] = consts.YMAKE_BUILD_SYSTEM

    def get_definition_flags(self):
        default_defenition_flags = sdk.parse_flags(' '.join(['-DALLOCATOR=J',
                                                                    '-DCFLAGS=-fno-omit-frame-pointer',
                                                                    '-DROBOT_BUILD',
                                                                    '-DROBOT_OLDTOK']))
        if self.ctx.get('enable_sanitizer', False):
            sanitizer_flags = sdk.parse_flags(' '.join(['-DSANITIZER_TYPE=address',
                                                               '-DPIRE_DEBUG']))
            default_defenition_flags = dict(default_defenition_flags, **sanitizer_flags)
        external_definition_flags = sdk.parse_flags(self.ctx.get(consts.DEFINITION_FLAGS_KEY, ''))
        def_flags = dict(default_defenition_flags, **external_definition_flags)
        def_flags["SANDBOX_TASK_ID"] = self.id
        return def_flags

    def on_execute(self):

        if self.do_pkg_tasks_exist():
            self.check_pkg_tasks()
            return

        self.ctx[consts.ARCADIA_URL_KEY] = self.normalize_svn_url(self.ctx[consts.ARCADIA_URL_KEY])

        os.putenv('LD_LIBRARY_PATH', '')
        BaseSimpleBuildTaskWithPkg.on_execute(self)

        self.save_resources_ids('ukrop.id.out.txt')

        if self.are_pkgs_enabled():
            self.mark_resources_ready()
            self.create_pkg_tasks(self.PKG_CTX)

    def on_release(self, additional_parameters):
        release_status = additional_parameters['release_status']
        z2_config = self.REL_STAT_TO_Z2_CFG_MAP.get(release_status)
        if not z2_config:
            return
        deploy_subtask_ctx = {
            'notify_via': 'email',
            'use_existing_pkgs': True,
            'build_task_id': self.id,
            'z2_config': z2_config,
        }
        deploy_subtask = self.create_subtask(
            task_type='DEPLOY_UKROP',
            input_parameters=deploy_subtask_ctx,
            description='Deploy task for task #{}: {}'.format(self.id, self.descr),
            priority=['SERVICE', 'HIGH']
        )
        self.ctx['deploy_subtask_id'] = deploy_subtask.id
        self.mark_released_resources(additional_parameters["release_status"])

    @staticmethod
    def normalize_svn_url(svn_url):
        parsed_url = Arcadia.parse_url(svn_url)
        svn_path = re.sub(r'/arcadia.*', '/arcadia', parsed_url.path, count=1)
        return Arcadia.replace(svn_url, path=svn_path)

    def save_resources_ids(self, ids_file):
        ids = []
        for resource_type, target, path in self.get_target_resources():
            field_name = target2field_name(target)
            if self.ctx[field_name]:
                id_key = '%s_resource_id' % target
                id_value = self.list_resources(resource_type)[0].id
                ids.append('"%s":"%s"' % (id_key, id_value))
        log_dir = get_logs_folder()
        with open(os.path.join(log_dir, ids_file), 'w') as ids_fd:
            ids_fd.write(','.join(ids))


__Task__ = BuildUkrop
