# -*- coding: utf-8 -*-

import logging
import re
import os
import shutil

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.svn import Arcadia


class BuildUkropTests(YaMakeTask):
    type = 'BUILD_UKROP_TESTS'

    client_tags = ctc.Tag.LINUX_PRECISE
    cores = 24

    def __init__(self, task_id=0):
        YaMakeTask.__init__(self, task_id)

    def initCtx(self):
        self.ctx['targets'] = "yweb/robot/ukrop/ci"
        YaMakeTask.initCtx(self)

    @staticmethod
    def normalize_svn_url(svn_url):
        path = Arcadia.parse_url(svn_url).path
        svn_path = re.sub(r'/arcadia.*', '/arcadia', path, count=1)
        return Arcadia.replace(svn_url, path=svn_path)

    def get_trunk_triggers(self, arcadia_triggers_path, local_triggers_path):
        """Replace local triggers(local_triggers_path) with
        triggers from arcadia(arcadia_triggers_path)"""
        logging.debug("Replacing triggers: %s" % local_triggers_path)
        if not os.path.exists(os.path.dirname(local_triggers_path)):
            logging.warning("no triggers dir: %s" % os.path.dirname(local_triggers_path))
            os.makedirs(os.path.dirname(local_triggers_path))
        if os.path.exists(local_triggers_path):
            os.remove(local_triggers_path)
        triggers_url = os.path.join("arcadia:/arc/trunk/arcadia", arcadia_triggers_path)
        Arcadia.export(triggers_url, local_triggers_path)

    def post_build(self, source_dir, output_dir, pack_dir):
        ci_dir = os.path.join(output_dir, "yweb/robot/ukrop/ci")
        shutil.move(os.path.join(ci_dir, "yweb"), os.path.join(ci_dir, "libs/"))
        shutil.move(os.path.join(ci_dir, "contrib"), os.path.join(ci_dir, "libs/"))
        shutil.move(os.path.join(ci_dir, "kernel"), os.path.join(ci_dir, "libs/"))
        shutil.move(os.path.join(ci_dir, "solomon"), os.path.join(ci_dir, "libs/"))

        config_dir = os.path.join(ci_dir, "libs/ukrop_conf")
        os.makedirs(config_dir)
        shutil.copytree(os.path.join(source_dir, "yweb/robot/ukrop/conf"), os.path.join(config_dir, 'conf'))
        os.makedirs(os.path.join(config_dir, "mnformulas"))
        Arcadia.export("arcadia:/arc/trunk/arcadia/yweb/robot/ukrop/mnformulas", os.path.join(config_dir, "mnformulas"))

        ukrop_triggers_path = "yweb/robot/ukrop/conf/conf-production/kiwi/triggers.pb.txt"
        kiwi_triggers_path = "yweb/robot/kiwi/conf/kiwi/triggers.pb.txt"
        self.get_trunk_triggers(kiwi_triggers_path, os.path.join(config_dir, "triggers.pb.txt"))
        self.get_trunk_triggers(ukrop_triggers_path, os.path.join(config_dir, "conf/conf-production/kiwi/triggers.pb.txt"))

        for item in os.listdir(ci_dir):
            if item != "libs" and item != "scripts" and item != "tests":
                if os.path.isdir(os.path.join(ci_dir, item)):
                    shutil.rmtree(os.path.join(ci_dir, item))
                else:
                    os.remove(os.path.join(ci_dir, item))

        sid = self.prepare_resource(ci_dir)
        log_path = get_logs_folder()
        with open(os.path.join(log_path, 'tests.id.out.txt'), "w") as fid:
            fid.write(sid)

    def prepare_resource(self, resource_dir):
        os.chdir(resource_dir)
        cmd = "tar -czvf %s/ukrop_tests.tgz ." % self.abs_path()
        self._subprocess(cmd, check=True, wait=True, log_prefix='build_tar')
        tar_dir = os.path.join(self.abs_path(), 'ukrop_tests.tgz')
        r = self._create_resource("ukrop", tar_dir, resource_types.KWUKROPTESTS, arch=self.arch, attrs={"tests": "ukrop"})
        return '"ukroptests_resource_id":"%s",' % r.id


__Task__ = BuildUkropTests
