# -*- coding: utf-8 -*-

import __builtin__

import os

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build.BaseSimpleBuildTaskWithPkg import BaseSimpleBuildTaskWithPkg
from sandbox.projects.common.arcadia import sdk


def target2field_name(target):
    return 'build_%s' % target


class BuildUzor(BaseSimpleBuildTaskWithPkg):
    type = 'BUILD_UZOR'

    client_tags = ~ctc.Tag.Group.LINUX | ctc.Tag.LINUX_PRECISE

    TARGET_RESOURCES = (
        (resource_types.UZOR_SERVER_BIN, 'robot/uzor/bin/uzor'),
    )

    PKG_CTX = {
        'repo_key_id': 'Ukrop Project (Yandex Crawler) <robot-ukrop-sandbox@yandex-team.ru>',
        'gpg_private_key': 'robot-ukrop-sandbox-gpg-private',
        'gpg_public_key': 'robot-ukrop-sandbox-gpg-public',
        'ssh_private_key': 'robot-ukrop-sandbox-ssh-private',
    }

    RESOURCE_TO_PKG_MAP = {
        'linux': {
            'UZOR_SERVER_BIN': 'packages/debian/uzor/yandex-search-uzor-server-bin',
        },
    }

    class UzorBinaries(parameters.SandboxInfoParameter):
        name = 'info_field'
        description = 'Uzor binaries'

    input_parameters = BaseSimpleBuildTaskWithPkg.input_parameters + [UzorBinaries]
    for _, path in TARGET_RESOURCES:
        target = os.path.basename(path)
        field_name = target2field_name(target)
        input_parameters.extend([__builtin__.type(
            field_name,
            (parameters.SandboxBoolParameter,),
            dict(name=field_name, description=target, default_value=True),
        )])

    def initCtx(self):
        BaseSimpleBuildTaskWithPkg.initCtx(self)
        self.ctx['build_system'] = consts.YMAKE_BUILD_SYSTEM

    def get_definition_flags(self):
        default_defenition_flags = sdk.parse_flags(' '.join(['-DALLOCATOR=J',
                                                                    '-DCFLAGS=-fno-omit-frame-pointer',
                                                                    '-DROBOT_BUILD',
                                                                    '-DROBOT_OLDTOK']))
        if self.ctx.get('enable_sanitizer', False):
            sanitizer_flags = sdk.parse_flags(' '.join(['-DSANITIZER_TYPE=address',
                                                               '-DPIRE_DEBUG']))
            default_defenition_flags = dict(default_defenition_flags, **sanitizer_flags)
        external_definition_flags = sdk.parse_flags(self.ctx.get(consts.DEFINITION_FLAGS_KEY, ''))
        def_flags = dict(default_defenition_flags, **external_definition_flags)
        def_flags["SANDBOX_TASK_ID"] = self.id
        return def_flags

    def do_execute(self):
        if self.do_pkg_tasks_exist():
            self.check_pkg_tasks()
            return

        BaseSimpleBuildTaskWithPkg.do_execute(self)

        self.save_resources_ids('uzor.id.out.txt')

        if self.are_pkgs_enabled():
            self.mark_resources_ready()
            self.create_pkg_tasks(self.PKG_CTX)

    def save_resources_ids(self, ids_file):
        ids = []
        for resource_type, target, path in self.get_target_resources():
            field_name = target2field_name(target)
            if self.ctx[field_name]:
                id_key = '%s_resource_id' % target
                id_value = self.list_resources(resource_type)[0].id
                ids.append('"%s":"%s"' % (id_key, id_value))
        log_dir = get_logs_folder()
        with open(os.path.join(log_dir, ids_file), 'w') as ids_fd:
            ids_fd.write(','.join(ids))


__Task__ = BuildUzor
