# -*- coding: utf-8 -*-

import os
import shutil

from sandbox.projects import resource_types
from sandbox.projects.common.build.CommonBuildTask import CommonBuildTask
from sandbox.projects.resource_types.releasers import video as video_releasers
import sandbox.projects.common.constants as consts

from sandbox.sdk2.resource import AbstractResource
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.nanny import nanny


class VIDEO_SAAS_SEARCH_CONFIG(AbstractResource):
    """
        Файл конфига SAAS поиска видео для быстрого индекса
    """
    releasable = True
    auto_backup = True
    releasers = video_releasers


class VIDEO_RTY_SEARCH_CONFIG(AbstractResource):
    """
        Файл конфига SAAS RTY поиска видео для быстрого индекса
    """
    releasable = True
    auto_backup = True
    releasers = video_releasers


class VIDEO_RTY_EXP_SEARCH_CONFIG(AbstractResource):
    """
        Файл конфига SAAS RTY experiment поиска видео для быстрого индекса
    """
    releasable = True
    auto_backup = True
    releasers = video_releasers


class VIDEO_OXYGEN_OPTIONS_CONFIG(AbstractResource):
    """
        Файл конфига OxygenOptions.cfg поиска видео для быстрого индекса
    """
    releasable = True
    auto_backup = True
    releasers = video_releasers


class VIDEOQUICK_CONFIGS_BUNDLE(AbstractResource):
    """
        Quick rtyserver config bundle
    """
    releasable = True
    auto_backup = True
    releasers = video_releasers


class VHS_CONFIGS_BUNDLE(AbstractResource):
    """
        VHS rtyserver config bundle
    """
    releasable = True
    auto_backup = True
    releasers = video_releasers


class VideoRTYBuild(parameters.SandboxBoolParameter):
    name = 'build_videortyserver'
    description = 'Enable VideoRTYServer build'
    default_value = True


class BuildVideosearchBundle(nanny.ReleaseToNannyTask, CommonBuildTask):
    """
        Build videoseach bundle (basesearch, config) for all architectures.
    """

    type = 'BUILD_VIDEOSEARCH_BUNDLE'

    execution_space = 70000
    archs_for_bundle = ('linux', )
    input_parameters = CommonBuildTask.input_parameters + [VideoRTYBuild]

    TARGET_RESOURCE_TYPES = (
        resource_types.VIDEOSEARCH_EXECUTABLE,
    )

    def initCtx(self):
        CommonBuildTask.initCtx(self)
        self.ctx[consts.BUILD_SYSTEM_KEY] = consts.YMAKE_BUILD_SYSTEM
        self.ctx['build_bundle'] = False

    def get_config_dir(self):
        return self.abs_path('config')

    def on_enqueue(self):
        CommonBuildTask.on_enqueue(self)

        self._create_resource('Video main search config for "%s"' % self.descr,
                              os.path.join(self.get_config_dir(), 'VideoTier0.cfg'),
                              resource_types.VIDEO_SEARCH_CONFIG,
                              arch='any')
        self._create_resource('Video SSD search config for "%s"' % self.descr,
                              os.path.join(self.get_config_dir(), 'VideoTier0-ssd.cfg'),
                              resource_types.VIDEO_SEARCH_SSD_CONFIG,
                              arch='any')
        self._create_resource('Videoquick config bundle for "%s"' % self.descr,
                              os.path.join(self.get_config_dir(), 'videoquick'),
                              VIDEOQUICK_CONFIGS_BUNDLE,
                              arch='any')
        self._create_resource('VHS config bundle for "%s"' % self.descr,
                              os.path.join(self.get_config_dir(), 'vhs'),
                              VHS_CONFIGS_BUNDLE,
                              arch='any')

    def bundle_configs(self, source_dir, target_dir, bundled_files=None):
        if not bundled_files:
            bundled_files = ['basesearch-refresh', 'OxygenOptions.cfg', 'rtyserver.conf-common']
        if not os.path.exists(target_dir):
            os.makedirs(target_dir)

        for file in bundled_files:
            source_file = os.path.join(source_dir, file)
            target_file = os.path.join(target_dir, file)
            shutil.copyfile(source_file, target_file)

    def on_execute(self):
        self.set_info('Building binaries')
        CommonBuildTask.on_execute(self)

        self.set_info('Making config')
        os.chdir(self.abs_path())       # To avoid finding ourselves in a deleted dir

        config_dir_in_arcadia = os.path.join(os.path.dirname(
            resource_types.VIDEOSEARCH_EXECUTABLE.arcadia_build_path), 'config')
        make_folder(self.get_config_dir(), True)
        config_dir_svn = Arcadia.append(self.ctx[consts.ARCADIA_URL_KEY], config_dir_in_arcadia)
        Arcadia.export(config_dir_svn, self.get_config_dir())
        self.set_info('All done')

    def get_target_resources(self):
        result = CommonBuildTask.get_target_resources(self)

        if self.ctx[VideoRTYBuild.name]:
            resource_type = resource_types.VIDEORTYSERVER_EXECUTABLE
            arcadia_build_name = getattr(resource_type, 'arcadia_build_name', self.get_target_name(resource_type.arcadia_build_path))
            result.append((resource_type, arcadia_build_name, resource_type.arcadia_build_path))

        return result


__Task__ = BuildVideosearchBundle
