import os
import platform

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types


class DebianPackageListParameter(parameters.SandboxStringParameter):
    name = "debian_package_list"
    description = "Space-separated .deb packages list to be installed"


class PythonPackageListParameter(parameters.SandboxStringParameter):
    name = "python_package_list"
    description = "Python packages list (if necessary pass version here: pack==v.vv.vvv)"


class WheelListParameter(parameters.SandboxStringParameter):
    name = 'wheels_list'
    description = 'Wheel packages list to install'


class BuildVirtualEnv(task.SandboxTask):
    """
    Build virtual environment with deb/python-packages and wheels
    """

    input_parameters = (
        DebianPackageListParameter,
        PythonPackageListParameter,
        WheelListParameter
    )
    type = "BUILD_VIRTUALENV"
    resource_path = "virtualenv.tar.gz"
    resource_type = resource_types.VIRTUALENV

    @property
    def privileged(self):
        return bool(self.ctx.get(DebianPackageListParameter.name, "").strip())

    @staticmethod
    def _patch_activate_this_py(venv):
        activate_this_py = os.path.join(venv.root_dir, "bin", "activate_this.py")
        contents = open(activate_this_py, "r").read()
        contents = contents.replace("sys.real_prefix = sys.prefix",
                                    "sys.real_prefix = getattr(sys, 'real_prefix', sys.prefix)")
        contents = contents.replace("old_os_path = os.environ['PATH']",
                                    "old_os_path = os.environ.get('PATH', '')")
        with open(activate_this_py, "w") as out:
            out.write(contents)

    def _install_debian_packages(self):
        debian_packages = self.ctx[DebianPackageListParameter.name]
        if debian_packages:
            process.run_process(["apt-get", "update"], log_prefix="apt-get")
            process.run_process(
                ["apt-get", "install", "-y"] + debian_packages.split(),
                log_prefix="apt-get"
            )

    def _install_wheels(self, venv):
        wheel_def = self.ctx.get(WheelListParameter.name, "")
        wheel_packages = list(tuple(dep.split("==")) for dep in wheel_def.split())
        for package, version in wheel_packages:
            environments.PipEnvironment(package, version, use_wheel=True, venv=venv).prepare()

    def _install_python_packages(self, venv):
        python_packages = self.ctx[PythonPackageListParameter.name]
        for package in python_packages.split():
            venv.pip(package)

        # requirements_path = self.ctx[PythonPackagesFileParameter.name]
        # venv.pip("-r %s" % requirements_path)

    def on_execute(self):
        self._install_debian_packages()

        with environments.VirtualEnvironment() as venv:
            venv.pip('-U pip')
            self._install_wheels(venv)
            self._install_python_packages(venv)
            self._patch_activate_this_py(venv)
            venv.pack_bundle(self.resource_path)

        self.create_resource(
            description=self.descr,
            resource_path=self.resource_path,
            resource_type=self.resource_type,
            attributes={"platform": platform.platform()},
        )


__Task__ = BuildVirtualEnv
