# -*- coding: utf-8 -*-

import copy
import logging
import os
import time
import urllib2

from sandbox.projects import resource_types

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common import apihelpers
import sandbox.projects.common.build.ArcadiaTask as bbt
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.wizard.wizard_builder import WizardBuilder
from sandbox.projects.common.wizard.svn_wizard import SvnWizard
from sandbox.projects.common.wizard.wizard_runner import WizardRunner
# from projects.common.wizard.parameters import NotRequiredWizardBuildField, NotRequiredWizardRuntimeField
from sandbox.projects.common.wizard.providers import WizardProvider
from sandbox.projects.common.wizard.utils import init_shard


WizardRuntimeSource = {
    "runtime_data": "wizard.source",
}


class BuildWizard(nanny.ReleaseToNannyTask, bbt.ArcadiaTask):
    """
    **Описание**
        Собирает бинарник визарда + тарбол с данными.
        В процессе работы запускает `BUILD_SEARCH <https://sandbox.yandex-team.ru/docs/html/tasks/projects.BuildSearch.html>`_, который и собирает визард.

    **Создаваемые ресурсы**

        * REMOTE_WIZARD - бинарник визарда
        * WIZARD_PACKAGE - тарбол с данными визарда
    """

    type = 'BUILD_WIZARD'

    execution_space = 102400

    class CloneId(parameters.SandboxStringParameter):
        name = 'clone_id'
        description = 'Clone other build task (empty to ignore)'

    class BuildRuntimeId(parameters.SandboxStringParameter):
        name = 'build_runtime_id'
        description = 'runtime build id (empty to create new, required if cloning)'

    class BuildSearchId(parameters.SandboxIntegerParameter):
        name = 'build_search_id'
        description = 'BUILD_SEARCH id (empty to create new)'

    class CloneShardResId(parameters.ResourceSelector):
        name = 'clone_shard_res_id'
        description = 'WIZARD_SHARD resource id to clone from (for clone only)'
        resource_type = resource_types.WIZARD_SHARD
        required = False

    class RuntimeData(parameters.SandboxStringParameter):
        name = 'runtime_data'
        description = 'runtime revision'

    class CheckWizard(parameters.SandboxBoolParameter):
        name = 'check_wizard'
        description = 'check wizard'
        default_value = True

    input_parameters = build_params.get_arcadia_params() + [
        CloneId,
        BuildRuntimeId,
        BuildSearchId,
        CloneShardResId,
        RuntimeData,
        CheckWizard
    ]

    additional_bundle_params = {'linux': {'is_worker': True}}

    def initCtx(self):
        self.ctx['shard_name'] = 'wizard-%s-%s' % (self.id, int(time.time()))
        if self.ctx.get('create_bsconfig_shard'):
            self.set_info('shard_name=%s' % self.ctx['shard_name'])

    def on_enqueue(self):
        bbt.ArcadiaTask.on_enqueue(self)
        if not self.ctx.get('is_worker'):
            for name, dest in WizardRuntimeSource.iteritems():
                url = "%s/%s" % ('arcadia:/robots/trunk/wizard-data', dest)
                rev = self.ctx.get(name)
                if not rev or str(rev).lower() == "head":
                    rev = Arcadia.get_revision(url)
                    self.ctx[name] = rev

    def test_wizard(self, arcadia_tests_data_dir):
        config_path = '%s/wizard.cfg' % self.abs_path()

        cmd = '%s/wizard/conf/config.py wizard > %s' % (arcadia_tests_data_dir, config_path)
        run_process(cmd.split(), shell=True)

        try:
            with WizardRunner(
                wizard_path='%s/binaries/wizard' % self.abs_path(),
                arcadia_tests_data_path=arcadia_tests_data_dir,
                config_path=config_path,
            ):
                try:
                    self.ctx['test_answer'] = urllib2.urlopen("http://localhost:8891/admin?action=version").read()
                except Exception as e:
                    logging.info("Failed to send a test request to wizard")
                    logging.exception(e)
        except Exception as e:
            logging.info("Failed to run wizard")
            logging.exception(e)
            raise SandboxTaskFailureError('Cant run wizard: %s' % e)

    def copy_resource(self, resource_id):
        resource = channel.sandbox.get_resource(resource_id)
        src_path = self.sync_resource(resource.id)
        dst_path = resource.file_name
        copy_path(src_path, dst_path)

        attributes = copy.copy(resource.attributes)
        attributes.pop('released', None)
        created_resource = self.create_resource(resource.description, dst_path, str(resource.type), arch=resource.arch,
                                                attributes=attributes)
        self.mark_resource_ready(created_resource)
        return created_resource

    def clone_resources_from_task(self, task_id, res_types=None):
        if not res_types:
            res_types = []
        for resource in apihelpers.list_task_resources(task_id):
            if resource.type in res_types:
                self.copy_resource(resource)

    def build(self, with_runtime=True):
        self.ctx['subs'] = []
        need_wait = False
        if self.ctx.get('build_search_id'):
            self.ctx['subs'].append(self.ctx['build_search_id'])
        else:
            build_ctx = copy.copy(self.ctx)
            build_ctx['notify_via'] = ''
            build_ctx['build_wizard'] = True
            build_ctx['build_printwzrd'] = True
            build_ctx['build_evlogdump'] = True
            build_ctx['build_system'] = 'ya'

            additional_binaries = ['gztcompiler', 'querydata_indexer',
                                   'music_wizard_data_converter', 'remorphc']
            for name in additional_binaries:
                build_ctx["build_%s" % name] = True

            subtask = self.create_subtask(task_type='BUILD_SEARCH', input_parameters=build_ctx,
                                          description='Wizard build for sandbox:%s' % self.id)
            self.ctx['build_search_id'] = subtask.id
            self.ctx['subs'].append(subtask.id)
            need_wait = True

        if with_runtime and not self.ctx.get('is_worker'):
            if not self.ctx.get('build_runtime_id'):
                self.ctx['build_runtime_id'] = WizardBuilder.build_runtime(
                    self.ctx['build_search_id'],
                    self.ctx.get('runtime_data'),
                )
                need_wait = True
            self.ctx['subs'].append(self.ctx['build_runtime_id'])

        if need_wait:
            self.wait_all_tasks_completed(self.ctx['subs'])

        # sync revisions with runtime build
        if with_runtime and self.ctx.get('build_runtime_id'):
            runtime_build_task = channel.sandbox.get_task(self.ctx['build_runtime_id'])
            for key in WizardRuntimeSource:
                self.ctx[key] = runtime_build_task.ctx[key]

        build_worker = channel.sandbox.get_task(self.ctx['subs'][0])
        self.ctx['arcadia_revision'] = build_worker.ctx['arcadia_revision']
        self.ctx['last_change'] = build_worker.ctx.get('last_change')

        self.copy_resource(WizardBuilder.wizard_from_task(self.ctx['subs'][0]))
        self.copy_resource(WizardBuilder.evlogdump_from_task(self.ctx['subs'][0]))
        self.copy_resource(WizardBuilder.printwzrd_from_task(self.ctx['subs'][0]))

        arcadia_tests_data_dir = self.abs_path('arcadia_tests_data')
        make_folder(arcadia_tests_data_dir, delete_content=True)

        if with_runtime and not self.ctx.get('is_worker'):
            wizard_runtime_package = self.copy_resource(WizardBuilder.runtime_package_from_task(self.ctx['build_runtime_id']))
            cmd = 'tar -C %s -xf %s' % (arcadia_tests_data_dir, wizard_runtime_package.path)
            run_process(cmd.split())

        data_attrs = self.get_data_shard_svn_attrs()
        if data_attrs:
            logging.info('Search for WIZARD_SHARD with attributes: {}'.format(data_attrs))
            data_resource = apihelpers.get_last_resource_with_attrs(resource_types.WIZARD_SHARD, data_attrs, all_attrs=True)
        else:
            data_resource = None

        if data_resource:
            wizard_data_local_path = self.copy_resource(data_resource).path
            os.symlink(wizard_data_local_path, os.path.join(arcadia_tests_data_dir, 'wizard'))
            self.ctx['shard_ready'] = True
        else:
            wizard_data_svn_path = SvnWizard(self.ctx[consts.ARCADIA_URL_KEY]).wizard_data()
            wizard_data_local_path = os.path.join(arcadia_tests_data_dir, 'wizard')
            Arcadia.export(wizard_data_svn_path, wizard_data_local_path)

        if not self.ctx.get('is_worker') and self.ctx.get('check_wizard'):
            self.test_wizard(arcadia_tests_data_dir)
        return wizard_data_local_path, os.path.join(arcadia_tests_data_dir, 'wizard.runtime')

    def get_data_shard_svn_attrs(self):
        wizard_data_svn_path = SvnWizard(self.ctx[consts.ARCADIA_URL_KEY]).wizard_data()
        wizard_data_svn_revision = Arcadia.info(wizard_data_svn_path)['commit_revision']
        if 'arc/trunk/' in wizard_data_svn_path:
            attributes = {
                'revision': wizard_data_svn_revision,
                'path': 'trunk',
            }
        elif 'arc/branches/wizard/test' in wizard_data_svn_path or 'arc/branches/junk' in wizard_data_svn_path:  # for test and debug purposes
            attributes = {
                'revision': wizard_data_svn_revision,
                'path': 'test',
            }
        else:
            attributes = {}
        return attributes

    def create_shard(self, wizardData):
        task_time_stamp = '%s-%s' % (self.id, int(time.time()))
        wizard_shard_name = 'wizard_shard-%s' % task_time_stamp

        wizard_shard_path = self.create_resource(
            description=self.descr,
            resource_path=wizard_shard_name,
            resource_type='WIZARD_SHARD',
            attributes=self.get_data_shard_svn_attrs(),
        ).path

        copy_path(wizardData, wizard_shard_path)

        configDir = os.path.join(wizard_shard_path, "conf")
        self._subprocess(
            "%s --all %s" % (os.path.join(configDir, "config.py"), configDir),
            wait=True,
            shell=True,
            log_prefix="config_generation"
        )
        copy_path(os.path.join(configDir, 'wizard.cfg'), 'wizard.cfg')
        cfg = self.create_resource('wizard config', 'wizard.cfg', resource_types.WIZARD_CONFIG)
        self.mark_resource_ready(cfg)

        # generating geosearch config
        thesaurus_config_dir = os.path.join(wizard_shard_path, "thesaurus")
        generator = os.path.join(thesaurus_config_dir, 'generate_geosearch_config.py')
        if os.path.exists(generator):
            soruce_config = os.path.join(thesaurus_config_dir, 'thesaurus.cfg')
            dest_config = os.path.join(thesaurus_config_dir, 'thesaurus_geo.cfg')
            self._subprocess(
                '{} < {} > {}'.format(generator, soruce_config, dest_config),
                wait=True,
                shell=True,
                log_prefix='thesaurus_geo_cfg_gen'
            )

        init_shard(self, wizard_shard_path)

    def clone(self, with_runtime=True):
        wizard_build = self.ctx['clone_id']
        runtime_build = self.ctx['build_runtime_id'] if with_runtime else None
        clone_shard_res_id = self.ctx['clone_shard_res_id']

        if 'wizard_cloned' not in self.ctx:
            wizard_task = channel.sandbox.get_task(wizard_build)
            if not wizard_task.is_finished():
                self.wait_task_completed(wizard_task)
            self.ctx[consts.ARCADIA_URL_KEY] = wizard_task.ctx[consts.ARCADIA_URL_KEY]
            wizard_resources = ['REMOTE_WIZARD', 'EVLOGDUMP_EXECUTABLE', 'WIZARD_CONFIG', 'PRINTWZRD']
            if clone_shard_res_id:
                self.copy_resource(clone_shard_res_id)
            else:
                wizard_resources.append('WIZARD_SHARD')
            self.clone_resources_from_task(wizard_build, wizard_resources)

            self.ctx['wizard_cloned'] = True

        if with_runtime and 'runtime_cloned' not in self.ctx:
            runtime_task = channel.sandbox.get_task(runtime_build)
            if not runtime_task.is_finished():
                self.wait_task_completed(runtime_task)
            for key in WizardRuntimeSource:
                self.ctx[key] = runtime_task.ctx[key]
            runtime_resources = ['WIZARD_RUNTIME_PACKAGE']
            self.clone_resources_from_task(runtime_build, runtime_resources)
            self.ctx['runtime_cloned'] = True

        with WizardProvider(wizard_build, runtime_build):
            try:
                self.ctx['test_answer'] = urllib2.urlopen("http://localhost:8891/wizard?action=printversion").read()
            except Exception:
                raise SandboxTaskFailureError("Failed to send a test request to wizard")

    def do_execute(self):
        if self.ctx.get('clone_id'):
            if not self.ctx.get('build_runtime_id'):
                raise SandboxTaskFailureError('cloning, but runtime build id is not provided')
            self.clone()
        else:
            wizard, runtime = self.build()
            if not self.ctx.get('is_worker') and not self.ctx.get('shard_ready'):
                self.create_shard(wizard)

    def _on_execute_helper(self):
        if self.arch == 'any':
            raise SandboxTaskFailureError('U should select arch, "any" is not supported')
        if self.ctx[consts.BUILD_BUNDLE_KEY] and 'linux' in self.arch:
            raise SandboxTaskFailureError('If you really need a bundle build, choose freebsd os')
        if self.ctx.get('clone_id'):
            self.ctx[consts.BUILD_BUNDLE_KEY] = False
        bbt.ArcadiaTask.on_execute(self)

    def on_execute(self):
        self._on_execute_helper()
        if self.is_build_bundle_task():
            param_src_task = channel.sandbox.get_task(self.ctx['freebsd_worker_task_id'])
            for key in WizardRuntimeSource:
                self.ctx[key] = param_src_task.ctx[key]


__Task__ = BuildWizard
