# -*- coding: utf-8 -*-
import logging
import os
import shutil
import sys
import json
import zipfile

from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.svn import zipatch
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common import decorators
from sandbox.projects.common.build import parameters as bp
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.wizard import resources as wr
from sandbox.projects.common.wizard import utils as wizard_utils


_CONFIG_TYPES = {
    resource_types.WIZARD_CONFIG:               'wizard-yaml.cfg',
    resource_types.WIZARD_GEO_CONFIG:           'geo-yaml.cfg',
    resource_types.WIZARD_ANTIROBOT_CONFIG:     'antirobot-yaml.cfg',
    resource_types.WIZARD_REALTIME_CONFIG:      'wizard_realtime.cfg',
    resource_types.WIZARD_RSYA_CONFIG:          'rsya-yaml.cfg',
    wr.WizardLingboostConfig:                   'lingboost.cfg',

    # Legacy 'NEW' resources. They are the same as the resources without this suffix.
    resource_types.WIZARD_CONFIG_NEW:           'new/wizard-yaml.cfg',
    resource_types.WIZARD_GEO_CONFIG_NEW:       'new/geo-yaml.cfg',
    resource_types.WIZARD_ANTIROBOT_CONFIG_NEW: 'new/antirobot-yaml.cfg',
    resource_types.WIZARD_REALTIME_CONFIG_NEW:  'new/wizard_realtime.cfg',
    resource_types.WIZARD_RSYA_CONFIG_NEW:      'new/rsya-yaml.cfg',
}

logger = logging.getLogger(__name__)


class Zipatch2(zipatch.Zipatch):
    def apply(self, dirpath, strip_prefix=""):
        strip_prefix = os.path.normpath(strip_prefix) + os.path.sep
        logging.debug("applying zipatch {f} to directory {d}, strip_prefix={p}".format(f=self.path, d=dirpath, p=strip_prefix))
        z = zipfile.ZipFile(self.path, "r")
        assert z.testzip() is None

        actions_json = z.read(zipatch.ACTIONS_FILE)
        actions = json.loads(actions_json)
        logging.debug("read {l} actions".format(l=len(actions)))
        for a in actions:
            logging.debug("applying action {a}".format(a=a))
            handler = zipatch.ACTION_HANDLERS[a["type"]]
            original_path = os.path.normpath(a["path"])
            if strip_prefix and not original_path.startswith(strip_prefix):
                logging.info("skipped path with wrong prefix {p}".format(p=original_path))
                continue
            relative_path = original_path
            if strip_prefix:
                relative_path = original_path[len(strip_prefix):]
                logging.info("original path: {o}, stripped path: {s}".format(o=original_path, s=relative_path))
            real_path = os.path.join(dirpath, relative_path)
            data = z.read(a["file"]) if a.get("file") else None
            handler(real_path, data)

        z.close()


class Arcadia2(Arcadia):
    @staticmethod
    def apply_patch_file(arcadia_path, arcadia_patch_path, is_zipatch=False, strip_prefix=""):
        """
            Apply patch prepared via fetch_patch
        """
        logger.info('Apply arcadia patch')
        if is_zipatch:
            z = Zipatch2(arcadia_patch_path)
            z.apply(arcadia_path, strip_prefix)
        else:
            from sandbox.sandboxsdk import process
            strip = len(strip_prefix.split(os.path.sep))
            process.run_process(
                [
                    'patch',
                    '-d', arcadia_path,
                    '--ignore-whitespace',
                    '-p%d' % strip,
                    '-i', arcadia_patch_path,
                ],
                log_prefix='patch_arcadia'
            )

        return arcadia_patch_path  # i wonder if this file is gonna be ever deleted

    @classmethod
    def apply_patch(cls, arcadia_path, arcadia_patch, dest_dir, strip_prefix=""):
        """
            Apply patch from torrent or as text diff
            (diff should be taken from arcadia root)
        """
        arcadia_patch_path, is_zipatch = cls.fetch_patch(arcadia_patch, dest_dir)
        if arcadia_patch_path is None:
            return None
        else:
            return cls.apply_patch_file(arcadia_path, arcadia_patch_path, is_zipatch, strip_prefix)


class ConfigTypes(parameters.SandboxBoolGroupParameter):
    name = 'config_types_parameter'
    choices = [(str(k), str(k)) for k in _CONFIG_TYPES]
    description = 'Config types to build'
    default_value = str(resource_types.WIZARD_CONFIG)


class IsTestingConfig(parameters.SandboxBoolParameter):
    '''
    Use testing misspell hosts
    '''
    name = 'testing_config'
    description = 'Is testing config'


class BuildPrintwzrdConfig(parameters.SandboxBoolParameter):
    name = 'build_printwzrd'
    description = 'Also build all configs for printwizard tests'
    output_path = 'printwizard/'
    resource_type = resource_types.PRINTWZRD_CONFIG
    default_value = False


class NoCache(parameters.SandboxBoolParameter):
    name = 'no_cache'
    description = 'Disable WizardCache'
    default_value = False


class ParentResources(parameters.DictRepeater, parameters.SandboxStringParameter):
    # This parameter is used by BUILD_WIZARD_2 and is not visible in the UI.
    name = 'parent_resources'
    description = 'Transfer resources to parent task (resource_type, resource_id)'


class BuildWizardConfig(bugbanner.BugBannerTask, task.SandboxTask):
    type = 'BUILD_WIZARD_CONFIG'
    execution_space = 500
    input_parameters = [
        bp.ArcadiaUrl,
        bp.ArcadiaPatch,
        ConfigTypes,
        IsTestingConfig,
        BuildPrintwzrdConfig,
        NoCache,
    ]

    client_tags = wizard_utils.ALL_SANDBOX_HOSTS_TAGS

    def on_enqueue(self):
        task.SandboxTask.on_enqueue(self)
        types = utils.get_or_default(self.ctx, ConfigTypes)
        if not types and not utils.get_or_default(self.ctx, BuildPrintwzrdConfig):
            raise errors.SandboxTaskFailureError('no targets to build')

        self.ctx['resources'] = {}

        for res_type in types.split():
            if res_type not in _CONFIG_TYPES:
                raise errors.SandboxTaskFailureError('{} is not a Wizard config resource'.format(res_type))
            res = self.create_resource(self.descr, 'conf/{}'.format(_CONFIG_TYPES[res_type]), res_type)
            self.ctx['resources'][res_type] = res.id

        if utils.get_or_default(self.ctx, BuildPrintwzrdConfig):
            self.create_resource(self.descr, BuildPrintwzrdConfig.output_path, BuildPrintwzrdConfig.resource_type)

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.Wizard)
        conf_src_path = self._svn_checkout(resource_types.WIZARD_CONFIG.arcadia_build_path)
        output_dir = 'conf'
        paths.make_folder(output_dir, delete_content=True)
        generate = [sys.executable, os.path.join(conf_src_path, 'config.py')]
        argv = generate + ['--all', output_dir]
        if self.ctx.get(IsTestingConfig.name):
            argv.append('--testing')
        if self.ctx.get(NoCache.name):
            argv.append('--no-cache')
        process.run_process(argv, log_prefix='config_generation', wait=True, check=True)
        shutil.copytree('conf', 'conf/new')

        try:
            argv = generate + ['--enable-lingboost']
            with open('conf/{}'.format(_CONFIG_TYPES[wr.WizardLingboostConfig]), 'w') as cfg:
                process.run_process(argv, log_prefix='lingboost', wait=True, check=True, stdout=cfg, outputs_to_one_file=False)
        except Exception:
            # old configs are always suitable for lingboost
            with open('conf/{}'.format(_CONFIG_TYPES[wr.WizardLingboostConfig]), 'w') as cfg:
                process.run_process(generate, log_prefix='lingboost', wait=True, check=True, stdout=cfg, outputs_to_one_file=False)

        for res_type, res_id in (self.ctx.get(ParentResources.name) or {}).iteritems():
            path = channel.sandbox.get_resource(self.ctx['resources'][res_type]).path
            self.save_parent_task_resource(path, int(res_id))

        if utils.get_or_default(self.ctx, BuildPrintwzrdConfig):
            argv = generate + ['--printwizard', BuildPrintwzrdConfig.output_path]
            process.run_process(argv, log_prefix='printwzrd_config_generation', wait=True, check=True)

    @decorators.retries(3, delay=5, raise_class=errors.SandboxTaskUnknownError)
    def _svn_checkout(self, source):
        '''
        return: string, path to the patched dir 'source' inside Arcadia
        '''
        url = utils.get_or_default(self.ctx, bp.ArcadiaUrl)
        arc = Arcadia.parse_url(url)
        arc = Arcadia.replace(url, path=os.path.join(arc.path, source))
        arcadia_src_dir = Arcadia.get_arcadia_src_dir(arc)
        path_to_skip = os.path.normpath(resource_types.WIZARD_CONFIG.arcadia_build_path)
        Arcadia2.apply_patch(arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path(), strip_prefix=path_to_skip)
        return arcadia_src_dir


__Task__ = BuildWizardConfig
