# -*- coding: utf-8 -*-

import os

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel

import sandbox.projects.common.build.parameters as build_params
import sandbox.sandboxsdk.parameters as sb_params

from sandbox.projects import resource_types
from sandbox.projects.common import constants, utils
from sandbox.projects.common.wizard import utils as wizard_utils
from sandbox.projects.common.build.YaMake import YaMakeTask

import sandbox.projects.websearch.begemot.resources as br


def execute_splitting(binary_path, stripped_path, debug_path):
    args_1 = [
        'objcopy',
        '--only-keep-debug', binary_path,
        debug_path,
    ]

    args_2 = [
        'objcopy',
        '--strip-debug',
        '--remove-section', '.gnu_debuglink',
        '--add-gnu-debuglink', debug_path,
        binary_path,
        stripped_path,
    ]

    run_process(args_1, log_prefix="objcopy")
    run_process(args_2, log_prefix="objcopy")


class UseSelectiveCheckout(sb_params.SandboxBoolParameter):
    name = 'SelectiveCheckout'
    description = 'Selective Checkout (SANDBOX-4364)'
    default_value = False
    group = 'SVN Selective checkout'


class UseSshMultiplexing(sb_params.SandboxBoolParameter):
    name = 'SshMultiplexing'
    description = 'Enable SSH Multiplexing (SANDBOX-4364)'
    default_value = False
    group = 'SVN Selective checkout'


class BuildWizardExecutable(YaMakeTask):

    type = 'BUILD_WIZARD_EXECUTABLE'

    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.ArcadiaPatch,
        build_params.BuildSystem,
        build_params.BuildType,
        build_params.Sanitize,
        build_params.SeparateDebug,
        build_params.YtStore,
        UseSelectiveCheckout,
        UseSshMultiplexing,
    ]
    # basically wizard needs around 47+ Gb (or even more)
    # TODO: reduce to 40 GB once selective checkout is enabled again (DEVTOOLS-3629)
    execution_space = 120 * 1024  # holy crap, HDD limit exceeded
    ram = 50  # otherwise clang sometimes gets killed
    TARGET_RESOURCE_TYPES = [
        resource_types.EVLOGDUMP_EXECUTABLE,
        resource_types.MUSIC_WIZARD_DATA_CONVERTER,
        resource_types.PRINTWZRD,
        resource_types.REMORPH_DATA_COMPILER,
        br.WIZARD_FULL_EXECUTABLE,
        resource_types.TRIECOMPILER_EXECUTABLE,
        resource_types.UNPACKRICHTREE,
        resource_types.UNPACKREQBUNDLE,
        resource_types.WIZARD_DATA_COMPILER,
        resource_types.WIZARDYT,
    ]

    # Here we need to list all possible tags and later refine them in on_enqueue.
    # That's the only possible way in SDK1 (and YaMakeTask is currently on SDK1).
    client_tags = wizard_utils.ALL_SANDBOX_HOSTS_TAGS
    cores = 24

    def get_resources(self):
        resources = {}
        for rtype in self.TARGET_RESOURCE_TYPES:
            resource = {
                'description': rtype.name,
                'resource_type': rtype,
                'resource_path': getattr(rtype, 'build_output_dir', os.path.basename(rtype.arcadia_build_path)),
            }
            resources[rtype.name] = resource
        return resources

    def initCtx(self):
        self.ctx['kill_timeout'] = 6 * 60 * 60
        self.ctx['aapi_fallback'] = True  # DEVTOOLS-4664
        self.ctx['use_aapi_fuse'] = True  # BEGEMOT-1971
        YaMakeTask.initCtx(self)

    def on_enqueue(self):
        if self.ctx.get('testenv_database') == 'ws-begemot-trunk':
            self.execution_space = 85 * 1024  # we almost always hit cache, HDD exceeds are unlikely
        wizard_utils.setup_hosts_sdk1(self)
        YaMakeTask.on_enqueue(self)

    def pre_execute(self):
        # if not self.ctx.get(build_params.ArcadiaPatch.name):
        #     self.ctx[constants.CHECKOUT] = True
        # TODO Enable back selective checkout once it's not as painfully slow (DEVTOOLS-3629)
        self.ctx[constants.CHECKOUT] = self.ctx.get(UseSelectiveCheckout.name, False)
        if self.ctx.get(UseSshMultiplexing.name, False):
            self.ctx[constants.MULTIPLEX_SSH] = True

        YaMakeTask.pre_execute(self)

    def get_targets(self):
        return [
            os.path.dirname(resource_type.arcadia_build_path) for resource_type in self.TARGET_RESOURCE_TYPES
        ]

    def get_arts(self):
        return [
            {
                'path': resource_type.arcadia_build_path,
                'dest': getattr(resource_type, 'build_output_dir', ''),
            } for resource_type in self.TARGET_RESOURCE_TYPES
        ]

    def get_build_def_flags(self):
        return '-DSANDBOX_TASK_ID={}'.format(self.id)

    def post_build(self, source_dir, output_dir, pack_dir):
        if utils.get_or_default(self.ctx, build_params.SeparateDebug):
            self.split_binary(pack_dir)

        for resource_type in self.TARGET_RESOURCE_TYPES + [br.WIZARD_DEBUG, resource_types.REMOTE_WIZARD]:
            resources = channel.sandbox.list_resources(resource_type=resource_type, task_id=self.id) or []
            for resource in resources:
                if resource_type in [resource_types.WIZARDYT]:
                    channel.sandbox.set_resource_attribute(resource_id=resource.id, attribute_name='ttl', attribute_value=30)
                # Проверям передавал ли родительский таск id ресурса, который нужно заполнить
                if self.ctx.get(resource_type.name):
                    self.save_parent_task_resource(resource.path, self.ctx[resource_type.name])

        wizard_utils.delete_raw_build_output(self.id)

    def split_binary(self, pack_dir):
        # Splits executable binary into pure binary and debug info
        # return: path to debug info
        binary_path = os.path.join(pack_dir, self.get_resources()[br.WIZARD_FULL_EXECUTABLE.name]['resource_path'])
        stripped_path = os.path.join(pack_dir, 'wizard.stripped')
        debug_path = os.path.join(pack_dir, 'wizard.debug')

        execute_splitting(binary_path, stripped_path, debug_path)

        self.create_resource(
            'wizard service stripped executable',
            stripped_path,
            resource_types.REMOTE_WIZARD,
        )
        self.create_resource(
            'wizard debug info from binary',
            debug_path,
            br.WIZARD_DEBUG,
        )
        return debug_path


__Task__ = BuildWizardExecutable
