import os

from sandbox import sdk2

from sandbox.projects.common.yappy.base_build_python_bundle import BaseBuildPythonBundle


class BuildYDL(BaseBuildPythonBundle):
    @property
    def github_context(self):
        return '[Sandbox] Build and test'

    @property
    def bundle_description(self):
        return 'YDL bundle'

    @property
    def project_conf(self):
        return {
            'github': {
                'report_statuses': self.Parameters.report_github_statuses
            }
        }

    @staticmethod
    def parse_django_settings(settings_diff):
        settings = {}
        for line in settings_diff.split('\n'):
            splitted = line.split(' = ')
            if len(splitted) == 1:
                settings[line] = None
            else:
                key = splitted[0].strip()
                value = ' = '.join(splitted[1:])

                # Remove quotes.
                value = value.strip()
                if value.startswith("'") and value.endswith("'"):
                    value = value[1:-1]

                settings[key] = value

        return settings

    def after_install(self):
        with self.info_section('<install> static files'), sdk2.helpers.ProcessLog(self, 'install_static') as pl:
            self.run_command(
                [
                    self.Context.python_bin,
                    os.path.join(self.Context.sources_path, 'src', 'manage.py'), 'collectstatic',
                    '--settings', 'ydl.settings.build',
                    '--noinput',
                ],
                process_log=pl
            )

    def before_build(self):
        with self.info_section('<test> target'), sdk2.helpers.ProcessLog(self, 'test_target') as pl:
            self.run_command(
                [
                    self.Context.python_bin,
                    os.path.join(self.Context.sources_path, 'src', 'manage.py'), 'test',
                    '--settings', 'ydl.settings.build',
                    '--keepdb',
                    os.path.join(self.Context.sources_path, 'tests')
                ],
                process_log=pl
            )

    def after_build(self, **kwargs):
        with self.info_section('<build> info files'):
            info_files_dir = str(self.path('info_files'))
            os.mkdir(info_files_dir)

            version_file_path = os.path.join(info_files_dir, 'VERSION')
            revision_file_path = os.path.join(info_files_dir, 'revision.txt')
            task_id_file_path = os.path.join(info_files_dir, 'sandbox_task_id.txt')

            with sdk2.helpers.ProcessLog(self, 'settings_diff') as pl:
                settings_diff = self.run_command(
                    [
                        self.Context.python_bin,
                        os.path.join(self.Context.sources_path, 'src', 'manage.py'), 'diffsettings',
                        '--settings', 'ydl.settings.build',
                    ],
                    process_log=pl,
                    check_output=True
                )

            django_settings = self.parse_django_settings(settings_diff)
            static_files_path = django_settings.get(
                'STATIC_ROOT',
                os.path.join(self.Context.sources_path, 'collected_static')
            )

            with open(os.path.join(self.Context.sources_path, 'VERSION'), 'r') as source:
                version = source.read().strip().replace('\n', '')
                with open(version_file_path, 'w') as target:
                    target.write(version)

            with open(revision_file_path, 'w') as target:
                target.write('{}@{}'.format(
                    self.Parameters.project_git_base_ref, self.Parameters.project_git_base_commit
                ))

            with open(task_id_file_path, 'w') as target:
                target.write(str(self.id))

        additional_bundled_files = (
            (version_file_path, 'VERSION'),
            (revision_file_path, 'revision.txt'),
            (static_files_path, 'collected_static'),
            (task_id_file_path, 'sandbox_task_id.txt'),
        )

        super(BuildYDL, self).after_build(additional_bundled_files=additional_bundled_files)
