"""Build YabsYTStat (SBYT) packages."""
import logging

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.build import YaPackage
from sandbox.projects.common.build import parameters as build_parameters


class BuildYabsYTStat(YaPackage.YaPackage):
    type = 'BUILD_YABS_YTSTAT'

    class YabsYTStatPackagesParameter(YaPackage.PackagesParameter):
        default_value = None

    class AutoVersion(parameters.SandboxBoolParameter):
        name = 'auto_version'
        description = 'Set an automatic version?'

    input_parameters = (
        parameters.SandboxArcadiaUrlParameter,
        YabsYTStatPackagesParameter,
        AutoVersion,
    )

    OVERRIDDEN_CONTEXT = {
        YaPackage.PackageTypeParameter.name: YaPackage.TARBALL,
        YaPackage.StripBinariesParameter.name: True,
        # YaPackage.RunTestsParameter.name: True,
        # YaPackage.RunLongTestsParameter.name: True,
        YaPackage.UseNewFormatParameter.name: True,
        build_parameters.CheckoutParameter.name: True,
        build_parameters.UseArcadiaApiFuse.name: True,
        build_parameters.AllowArcadiaApiFallback.name: True,
    }

    DEFAULT_RESOURCE_TYPE = resource_types.YABS_YTSTAT_SANDBOX.name

    DEV_VERSION_PREFIX = '000.'

    def _populate_context(self):
        """Populate context using YaPackage defaults and our defaults."""
        input_parameters_names = set(parameter.name for parameter in self.input_parameters)
        for parameter in YaPackage.YaPackage.input_parameters:
            if parameter.name not in input_parameters_names:
                self.ctx[parameter.name] = parameter().default_value
        self.ctx.update(self.OVERRIDDEN_CONTEXT)
        self.ctx[YaPackage.ResourceTypeParameter.name] = self.DEFAULT_RESOURCE_TYPE

    def on_enqueue(self):
        self._populate_context()
        YaPackage.YaPackage.on_enqueue(self)

    def on_execute(self):
        YaPackage.YaPackage.on_execute(self)
        self.update_resources()

    def update_resources(self):
        """Do something with created resources."""
        resource_type = utils.get_or_default(self.ctx, YaPackage.ResourceTypeParameter)
        resources = self.list_resources(resource_type)
        for resource in resources:
            self.update_resource(resource)

    def update_resource(self, resource):
        """Set specific attributes for given resource.

        :type resource: yasandbox.proxy.resource.Resource
        """
        logging.info('Updating resource %s', resource)
        attrs = self.gen_attributes(resource)
        for name, value in attrs:
            logging.info('Setting attribute %s=%s', name, value)
            channel.channel.sandbox.set_resource_attribute(resource.id, name, value)

    def gen_attributes(self, resource):
        """Return an iterable of (name, value) attributes.

        Supposed to be overridden in subclasses.
        """
        if utils.get_or_default(self.ctx, self.AutoVersion):
            return [self._get_version_attr(resource)]
        else:
            return []

    def _get_version_attr(self, resource):
        return 'version', self.DEV_VERSION_PREFIX + resource.attrs['svn_revision']


__Task__ = BuildYabsYTStat
