"""Build YabsYTStat (SBYT) collectors."""
import itertools
import json
import logging
import os
import shutil
import tarfile
import tempfile

from sandbox.sandboxsdk import process

from sandbox.projects import BuildYabsYTStat
from sandbox.projects import resource_types
from sandbox.projects.common import utils


class BuildYabsYTStatCollector(BuildYabsYTStat.BuildYabsYTStat):
    type = 'BUILD_YABS_YTSTAT_COLLECTOR'

    DEFAULT_RESOURCE_TYPE = resource_types.YABS_YTSTAT_COLLECTOR.name

    COLLECTOR_ATTR_NAME = 'collector_{name}'
    COLLECTOR_ATTR_VALUE = '{log_type}:{binary}'

    def gen_attributes(self, resource):
        auto_version = utils.get_or_default(self.ctx, self.AutoVersion)
        return itertools.chain(
            [self._get_version_attr(resource)] if auto_version else [],
            [self._get_subtype_attr(resource)],
            self._get_collector_attrs(resource),
        )

    def _get_collector_attrs(self, resource):
        temp_dir = tempfile.mkdtemp()
        try:
            resource_path = resource.abs_path()
            with tarfile.open(resource_path, 'r:gz') as tar:
                tar.extractall(temp_dir)
            return self._gen_collector_attrs(temp_dir)
        finally:
            shutil.rmtree(temp_dir)

    def _get_subtype_attr(self, resource):
        packages_and_resources = self.ctx['_package_resources']
        for p, r in zip(packages_and_resources['packages'], packages_and_resources['resources']):
            if resource.id == r[0]:
                subtype = os.path.basename(os.path.dirname(p))
                return 'subtype', subtype

    def _gen_collector_attrs(self, temp_dir):
        attrs = []
        dest_tables = set()
        for file_name in os.listdir(temp_dir):
            file_path = os.path.join(temp_dir, file_name)

            if not os.path.isfile(file_path) or not os.access(file_path, os.X_OK):
                continue

            logging.info('Collector %s found', file_name)

            collector_process = process.run_process([file_path, 'list'], outs_to_pipe=True)
            out, err = collector_process.communicate()

            logging.debug('Collector info %r', err)

            collectors_info = json.loads(out)
            for collector_info in collectors_info:
                attr_name = self.COLLECTOR_ATTR_NAME.format(name=collector_info['name'])
                attr_value = self.COLLECTOR_ATTR_VALUE.format(
                    log_type=collector_info['log_type'],
                    binary=file_name,
                )
                attrs.append((attr_name, attr_value))
                dest_tables.update(collector_info.get('dest_tables', []))

        attrs.append(('dest_tables', ','.join(dest_tables)))

        return attrs


__Task__ = BuildYabsYTStatCollector
