# coding: utf-8

import os
import re
import lxml
import requests

import sandbox.common.types.client as ctc
from sandbox.common.types.misc import DnsType
from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.nanny.common.environments import GoDistEnvironment
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask


class DeployParameter(parameters.SandboxBoolParameter):
    name = 'deploy'
    description = 'Deploy new Yadi version (upload to MDS & Yadi service)'
    default_value = True


class BuildYadiTool(SandboxTask):
    type = 'BUILD_YADI_TOOL'

    dns = DnsType.DNS64  # github doesn't support ipv6 at the moment
    client_tags = ctc.Tag.LINUX_PRECISE
    environment = (
        GoDistEnvironment('1.8'),
    )
    input_parameters = [
        DeployParameter
    ]

    def on_execute(self):
        project_name = 'Yadi'
        clone_url = 'https://github.yandex-team.ru/product-security/yadi.git'
        ref_id = 'master'
        systems = ['linux', 'windows', 'darwin']
        package_path = self.path('yadi')
        self.git_clone_and_checkout(clone_url, package_path, ref_id)

        run_process(
            ['bash', './build.sh'] + systems,
            log_prefix='build',
            work_dir=package_path
        )

        version = ''
        with open(os.path.join(package_path, 'src/config/config.go'), 'r') as f:
            version = re.search(r'Version\s*=\s*"(?P<version>[^"]+)', f.read()).group('version')

        for system in systems:
            res = {
                'description': '%s v%s for %s' % (project_name, version, system.title()),
                'resource_type': resource_types.YADI_BINARY,
                'resource_path': self.pick_binary(package_path, system),
                'attributes': {'ttl': 'inf'}
            }
            if system == 'linux':
                res['arch'] = 'linux'

            self.create_resource(**res)
            self.deploy(res['resource_path'], version, system)

    def git_clone_and_checkout(self, package_url, package_path, ref_id):
        if not os.path.exists(package_path):
            run_process(
                ['git', 'clone', package_url, package_path],
                log_prefix='git clone',
                shell=True
            )
        else:
            run_process(
                ['git', 'fetch', 'origin'],
                work_dir=package_path,
                log_prefix='git fetch',
                shell=True
            )
        run_process(
            ['git', 'checkout', '-f', ref_id],
            work_dir=package_path,
            log_prefix='git checkout',
            shell=True
        )

    def pick_binary(self, package_path, system):
        target_path = os.path.join(package_path, '.bin', system)
        listing = os.listdir(target_path)
        if not listing:
            raise Exception('No binaries found in %s' % target_path)
        if len(listing) > 1:
            return target_path
        return os.path.join(target_path, listing[0])

    def deploy(self, binary_path, version, system):
        if not utils.get_or_default(self.ctx, DeployParameter):
            return

        name = os.path.basename(binary_path)
        mds_path = 'yadi-releases/%s/%s/%s' % (version, system, name)
        uploaded_key = self.mds_upload(binary_path, mds_path)

        auth = self.get_vault_data(self.owner, 'YADI_INTERNAL_TOKEN')
        headers = {
            'x-token': auth,
        }
        data = {
            'version': version,
            'system': system,
            'mds_key': uploaded_key,
            'stable': True
        }
        response = requests.post(
            'https://yadi.yandex-team.ru/api/v1/internal/release/new',
            headers=headers,
            json=data,
            verify=False  # Skynet, I look at you
        )

        if response.status_code != 200:
            raise Exception('Unable to release Yadi (code %d)' % response.status_code)

    def mds_upload(self, path, store_path):
        auth = self.get_vault_data(self.owner, 'YADI_MDS_TOKEN')

        headers = {
            'Authorization': 'Basic ' + auth,
        }

        with open(path, 'rb') as fp:
            response = requests.put(
                'http://storage-int.mdst.yandex.net:1111/upload-appchecker/' + store_path,
                headers=headers,
                data=fp
            )

        if response.status_code == 200:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            uploaded_key = tree.get('key')

            if uploaded_key:
                return uploaded_key

            return False
        elif response.status_code == 403:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            key = tree.find('key').text

            if key:
                raise Exception('Unable to upload to mds: key already exists: %s' % (key, ))
            else:
                raise Exception('Unable to upload to mds: got 403 without key')
        elif response.status_code == 401:
            raise Exception('Unable to upload to mds: perm denied (code 401)')
        else:
            raise Exception('Unable to upload to mds: code %d' % (response.status_code, ))


__Task__ = BuildYadiTool
