# -*- coding: utf-8 -*-

import os
import time
import json
import logging
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process

PACKAGE_NAME = 'frontend.tar.gz'
TESTING_PACKAGE_NAME = 'frontend-testing.tar.gz'
STATICS_SERVICE_ID = 'hot_nodejs_statics_prod'
NANNY_ACTIVATE_SNAPSHOT_TIME_LIMIT = 600
TAR_MODE = 'w:gz'


def unarchive(fname, newfname=None):
    if tarfile.is_tarfile(fname):
        if not newfname:
            idx = fname.find('.tar')
            newfname = fname[:idx]
        tar = tarfile.open(fname)
        tar.extractall(path=newfname)
        tar.close()
        os.unlink(fname)
        return newfname
    return fname


class NodeJSPackage(LastReleasedResource):
    name = 'YASAP_PDB_NODEJS_PACKAGE'
    description = 'YASAP_PDB_NODEJS_PACKAGE'
    resource_type = resource_types.YASAP_PDB_NODEJS_PACKAGE
    limit = 5


class GitBranch(SandboxStringParameter):
    name = "branch"
    description = "Branch"
    default_value = "master"


class Rebase(SandboxBoolParameter):
    name = "rebase"
    description = "rebase"
    default_value = False


class RebaseIgnore(SandboxBoolParameter):
    name = "rebase_ignore"
    description = "ignore rebase errors"
    default_value = False


class BuildHotFrontend(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_YASAP_HOT_FRONTEND'
    GIT_URL = 'https://github.yandex-team.ru/hot/hot-frontend.git'

    input_parameters = [NodeJSPackage, GitBranch, Rebase, RebaseIgnore]

    execution_space = 1200

    release_to = ['ftdebugger', "tarkus", "menjoy", "next0", "jakwuh"]

    def _create_tar(self, filenames, tarname=PACKAGE_NAME, tarmode=TAR_MODE):
        tar = tarfile.open(tarname, tarmode)
        for fname in filenames:
            logging.info('Add %s to archive', fname)
            tar.add(fname)
        tar.close()

    def _sync_res(self, resource):
        path = self.sync_resource(self.ctx[resource.name])
        copy_path(path, self.path())
        return path

    def _checkout_sources(self, tag, rebase, rebase_ignore):
        logging.info('Checking out %s tag of frontend', tag)
        src_dir = self.path('frontend')
        run_process(['git', 'clone', self.GIT_URL, src_dir], log_prefix='git_clone')
        run_process(['git', 'checkout', tag], work_dir=src_dir, log_prefix='git_checkout')

        try:
            if rebase:
                run_process(['git', 'config', '--global', 'user.email', 'polubob@yandex.ru'], work_dir=src_dir, log_prefix='git_config')
                run_process(['git', 'config', '--global', 'user.name', 'polubob'], work_dir=src_dir, log_prefix='git_config')
                run_process(['git', 'rebase', 'master'], work_dir=src_dir, log_prefix='git_rebase')
        except SandboxSubprocessError:
            if rebase_ignore:
                run_process(['git', 'rebase', '--abort'], work_dir=src_dir, log_prefix='git_rebase_abort')
            else:
                raise

        return src_dir

    def on_execute(self):
        files = []

        nodeArchive = self._sync_res(NodeJSPackage)
        nodeBaseName = os.path.basename(nodeArchive)
        nodeRoot = unarchive(nodeBaseName)
        node = os.path.abspath(os.path.join(nodeRoot, 'bin/node'))
        npm = os.path.abspath(os.path.join(nodeRoot, 'bin/npm'))
        binRoot = os.path.abspath(os.path.join(nodeRoot, 'bin'))

        sources_folder = self._checkout_sources(self.ctx['branch'], self.ctx['rebase'], self.ctx['rebase_ignore'])

        environment = os.environ.copy()
        environment['PATH'] = binRoot + ':' + os.environ['PATH']

        files.append(os.path.basename(nodeRoot))
        files.append(os.path.basename(sources_folder))

        run_process(
            [node, npm, '--registry=https://npm.yandex-team.ru', 'install', '--production'],
            log_prefix='npm_install',
            work_dir=sources_folder,
            environment=environment
        )

        # production

        run_process(
            [node, npm, 'run', 'release'],
            log_prefix='release',
            work_dir=sources_folder,
            environment=environment
        )

        self._create_tar(files, tarname=PACKAGE_NAME)

        self.create_resource(
            description='YASAP_HOT_FRONTEND_PACKAGE',
            resource_path=PACKAGE_NAME,
            resource_type=resource_types.YASAP_HOT_FRONTEND_PACKAGE,
            arch='any'
        )

        # testing

        run_process(
            [node, npm, 'run', 'release:testing'],
            log_prefix='release',
            work_dir=sources_folder,
            environment=environment
        )

        self._create_tar(files, tarname=TESTING_PACKAGE_NAME)

        self.create_resource(
            description='YASAP_HOT_FRONTEND_TESTING_PACKAGE',
            resource_path=TESTING_PACKAGE_NAME,
            resource_type=resource_types.YASAP_HOT_FRONTEND_TESTING_PACKAGE,
            arch='any'
        )

        pass

    def generate_service_resource(self):
        service_resource = {
            'task_id': str(self.id),
            'resource_type': 'YASAP_HOT_FRONTEND_PACKAGE',
            'task_type': self.type,
            'local_path': 'frontend-{}.tar.gz'.format(self.id)
        }
        return service_resource

    def get_snapshot_state(self, nanny_client, service_id, snapshot_id):
        current_state = nanny_client.get_service_current_state(service_id)
        active_snapshots = current_state['content']['active_snapshots']
        if not active_snapshots:
            ret_state = 'NO_SNAPSHOTS'
        elif active_snapshots[0]['snapshot_id'] != snapshot_id:
            ret_state = 'NOT_FOUND'
        else:
            ret_state = active_snapshots[0]['state']
        logging.debug('State of snapshot %s is %s', snapshot_id, ret_state)
        return ret_state

    def update_sandbox_files_on_release(self, sandbox_files, new_sandbox_file, max_count):
        def by_task_id_comparator(a, b):
            a_id = int(a['task_id'])
            b_id = int(b['task_id'])
            return a_id - b_id

        def by_task_type_filter(file):
            return file['task_type'] == self.type

        other_files = filter(lambda a: not by_task_type_filter(a), sandbox_files)
        needed_files = sorted(filter(by_task_type_filter, sandbox_files), by_task_id_comparator)
        needed_files = filter(lambda f: f['task_id'] != new_sandbox_file['task_id'], needed_files)
        needed_files.append(new_sandbox_file)
        return other_files + needed_files[-max_count:]

    def on_release(self, additional_parameters):
        service_id = STATICS_SERVICE_ID

        nanny_oauth_token = self.get_vault_data('YASAP', 'nanny_oauth_token')
        nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=nanny_oauth_token)

        service_resources = nanny_client.get_service_resources(service_id)

        logging.info('%s: Updating nanny service_resources', service_id)
        logging.debug('service_resources dump: %s', json.dumps(service_resources))

        new_service_resource = self.generate_service_resource()
        service_resources['content']['sandbox_files'] = self.update_sandbox_files_on_release(
            service_resources['content']['sandbox_files'],
            new_service_resource,
            6
        )

        logging.debug('service_resources dump: %s', json.dumps(service_resources))

        comment = 'Automatically add new resource: {}'.format(json.dumps(new_service_resource))
        logging.info('Generated comment: %s', comment)

        snapshot_info = nanny_client.update_service_resources(service_id, {
            'content': service_resources['content'],
            'comment': comment,
            'snapshot_id': service_resources['snapshot_id'],
        })

        logging.info('%s: nanny service was updated successfully', service_id)
        logging.debug('%s: snapshot_info %s', service_id, json.dumps(snapshot_info))

        logging.info('Setting new snapshot state to ACTIVE...')

        new_snapshot_id = snapshot_info['runtime_attrs']['_id']
        event_info = nanny_client.create_event(service_id, {
            'type': 'SET_SNAPSHOT_STATE',
            'content': {
                'state': 'ACTIVE',
                'recipe': 'common',
                'comment': comment,
                'snapshot_id': new_snapshot_id
            }
        })

        logging.debug('event_info dump: %s', json.dumps(event_info))
        logging.info('Now waiting for nanny service to make the new snapshot ACTIVE')

        time_limit = NANNY_ACTIVATE_SNAPSHOT_TIME_LIMIT
        start_time = time.time()
        while self.get_snapshot_state(nanny_client, service_id, new_snapshot_id) != 'ACTIVE':
            elapsed_time = int(round(time.time() - start_time))
            if elapsed_time > time_limit:
                raise Exception('Release failed due to time limit in {} s'.format(time_limit))

            logging.info('Waiting for new snapshot to become ACTIVE for %s s', elapsed_time)
            time.sleep(10)

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)

        logging.info('Resource released successfully!')


__Task__ = BuildHotFrontend
