# -*- coding: utf-8 -*-

import os
import logging
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process

ARCHIVE = 'iseg.tar.gz'
TAR_MODE = 'w:gz'


class BuildYasapIseg(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_YASAP_ISEG'
    GIT_URL = 'https://github.yandex-team.ru/specadmin/yandex-iseg-www.git'

    def _checkout_sources(self, tag):
        logging.info('Checking out %s tag', tag)
        src_dir = self.path('iseg')
        run_process(['git', 'clone', self.GIT_URL, src_dir], log_prefix='git_clone')
        run_process(['git', 'checkout', tag], work_dir=src_dir, log_prefix='git_checkout')
        return src_dir

    def on_execute(self):
        sources_folder = self._checkout_sources('master')
        src = os.path.abspath(os.path.join(sources_folder, 'yandex-iseg-www/src'))

        # Build archive
        tar = tarfile.open(ARCHIVE, TAR_MODE)
        logging.info('Add %s to archive', src)
        tar.add(src, arcname="src")
        tar.close()

        # Create resource
        self.create_resource(
            description='YASAP_ISEG_FRONTEND',
            resource_path=ARCHIVE,
            resource_type=resource_types.YASAP_ISEG_FRONTEND,
            arch='any'
        )

        pass

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        # release_status = additional_parameters.get('release_status')
        # release_subject = additional_parameters.get('release_subject')
        release_comments = additional_parameters.get('release_comments')
        service_id = "iseg_production"

        nanny_oauth_token = self.get_vault_data('YASAP', 'nanny_oauth_token')
        nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=nanny_oauth_token)

        nanny_client.update_service_sandbox_resources(
            service_id=service_id,
            task_type=self.type,
            task_id=str(self.id),
            comment=release_comments,
            deploy=True,
            deploy_comment=release_comments
        )


__Task__ = BuildYasapIseg
