import os
import glob
import shutil
import zipfile
import logging
import platform
import tempfile
import contextlib
import subprocess as sp

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.channel as sdk_channel
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.environments as sdk_environments
import sandbox.projects.resource_types

from sandbox import common
import sandbox.common.types.misc as ctm


@contextlib.contextmanager
def TmpDir(*args, **kwargs):

    path = tempfile.mkdtemp(*args, **kwargs)
    yield path
    shutil.rmtree(path)


def wheel_fix_rpath(whl_path):
    with TmpDir() as tmp_dir:
        with zipfile.ZipFile(whl_path) as whl_zip:
            whl_zip.extractall(tmp_dir)

        def so_walker(path):
            for root, dirs, files in os.walk(path):
                for f in files:
                    if f.endswith('.so'):
                        yield os.path.join(root, f)

        for fname in so_walker(tmp_dir):
            tmp_rel_path = os.path.relpath(fname, tmp_dir)
            try:
                logging.info("Updating library '%s'", tmp_rel_path)
                sp.check_output(
                    [
                        "/skynet/python/bin/chrpath",
                        "--replace",
                        "/skynet/python/lib",
                        fname
                    ],
                    stderr=sp.STDOUT
                )
            except sp.CalledProcessError as ex:
                if ex.returncode == 2:
                    logging.warning("Lib '%s' have no RPATH", tmp_rel_path)
                else:
                    logging.error("Fail updating dynlib '%s': %s", tmp_rel_path, ex.output)
                return False

        shutil.make_archive(whl_path, 'zip', tmp_dir)
    os.rename(whl_path + '.zip', whl_path)
    return True


class BuildPyPackage(sdk_task.BuildForAllMode):
    """
    Build .whl archive with python wheel of specified package for current platform
    """
    class PackageName(sdk_parameters.SandboxStringParameter):
        """Package name input parameter."""
        name = "package_name"
        description = "Python package name (if necessary pass version here: pack==v.vv.vvv)"

    class PreVersion(sdk_parameters.SandboxBoolParameter):
        """Boolean parameter to use or not `pre` version of package. False by default."""
        name = "use_pre_version"
        description = "Use `pre` version of package"

    class NoBinaryList(sdk_parameters.SandboxStringParameter):
        name = "no_binary_list"
        description = "package names with commas between them"
        default_value = ":all:"

    class NoBinary(sdk_parameters.SandboxBoolParameter):
        name = "no_binary"
        description = "Do not use binary packages (wheels)"
        default_value = True
    NoBinary.sub_fields = {'true': [NoBinaryList.name]}

    class FailIfBrokenRpath(sdk_parameters.SandboxBoolParameter):
        name = "fail_if_broken_rpath"
        description = "Fail task if can't fix RPATH"
        default_value = True

    class AdditionalParams(sdk_parameters.SandboxStringParameter):
        name = "additional_args"
        description = "Pass additional args for pip"
        default_value = ""

    class AdditionalPackages(sdk_parameters.SandboxStringParameter):
        name = "additional_packages"
        description = "Space-separated .deb packages list to be installed before build the wheel"
        default_value = ""

    class AdditionalPipPackages(sdk_parameters.SandboxStringParameter):
        name = "additional_pip_packages"
        description = "Space-separated pip packages list to be installed before build the wheel (pytest==2.5.1)"
        default_value = ""

    input_parameters = [
        PackageName,
        PreVersion,
        NoBinary,
        NoBinaryList,
        FailIfBrokenRpath,
        AdditionalParams,
        AdditionalPackages,
        AdditionalPipPackages
    ]

    type = "BUILD_YASAP_TEST_PY_PACKAGE"

    resource_type = str(sandbox.projects.resource_types.PYTHON_WHEEL)

    @property
    def privileged(self):
        return bool(self.ctx.get(self.AdditionalPackages.name, "").strip())

    @classmethod
    def required_resources(cls):
        return [(cls.resource_type, "Python wheel for {plat}".format(plat=platform.platform()))]

    @staticmethod
    def get_package_version(name, wheel_path):
        version = "unknown"
        try:
            import pip.wheel
        except ImportError as error:
            logging.error("Can not import pip.wheel package: \n %s", error)
            return name, version
        if "==" in name:
            return name.split("==")

        prefix = "{}-".format(name.lower().replace("-", "_"))
        for fname in os.path.os.listdir(wheel_path):
            if fname.lower().startswith(prefix):
                return name, pip.wheel.Wheel(fname).version

        return name, version

    def on_execute(self):
        if self.ctx.get(self.AdditionalPackages.name):
            sdk_process.run_process(["apt-get", "update"], log_prefix="apt-get")
            sdk_process.run_process(
                ["apt-get", "install", "-y"] + self.ctx[self.AdditionalPackages.name].split(),
                log_prefix="apt-get"
            )

        wheel_name = self.ctx[self.PackageName.name].strip()
        wheel_path = self.path("wheel")
        build_path = self.path("build")
        if not os.path.exists(wheel_path):
            logging.info("Creating {} path".format(wheel_path))
            os.mkdir(wheel_path)

        with sdk_environments.VirtualEnvironment(do_not_remove=True) as venv:
            venv.pip("pip")
            venv.pip("setuptools")

            # Since 0.26 it set SOABI tags to platform-specific wheels built for Python 2.X
            # (Pull Request #55, Issue #63, Issue #101)
            # which pip7 does not understand.
            # Can use newer version when upgrade pip to 8.xx.
            venv.pip("wheel==0.26.0")

            if self.ctx.get(self.AdditionalPipPackages.name):
                venv.pip(self.ctx[self.AdditionalPipPackages.name])

            pip = os.path.join(os.path.dirname(venv.executable), 'pip')
            env = os.environ.copy()

            if common.config.Registry().this.system.family in (ctm.OSFamily.LINUX,
                                                               ctm.OSFamily.LINUX_ARM):
                # Reserv space for RPATH
                # Set rpath in libraries, analogue LDFLAGS="-Wl,-R/skynet/python/lib"
                env['LD_RUN_PATH'] = ':' * 256
                env['LIBRARY_PATH'] = '/skynet/python/lib'
                env['CFLAGS'] = '-I/usr/local/include'
                env['LDFLAGS'] = '-L/skynet/python/lib -L/usr/local/lib'

            if common.config.Registry().this.system.family != ctm.OSFamily.OSX:
                env['LDFLAGS'] += ' -Wl,-R/skynet/python/lib'

            sdk_process.run_process([venv.executable, pip, 'list'], environment=env, log_prefix='pip_list')
            sdk_process.run_process(['env | sort'], environment=env, log_prefix='env', shell=True)

            cmd = [
                venv.executable,
                pip,
                "wheel",
                "--wheel-dir={}".format(wheel_path),
                "--build={}".format(build_path),
                "--index-url=https://pypi.yandex-team.ru/simple/",
                wheel_name
            ]

            if self.ctx.get(self.PreVersion.name):
                cmd.append("--pre")
            if self.ctx.get(self.AdditionalParams.name):
                cmd.append(self.ctx.get(self.AdditionalParams.name))
            if self.ctx[self.NoBinary.name]:
                cmd.append("--no-binary")
                cmd.append(self.ctx[self.NoBinaryList.name])

            sdk_process.run_process(cmd, environment=env, log_prefix='pip_wheel')

        if common.config.Registry().this.system.family in (ctm.OSFamily.LINUX,
                                                           ctm.OSFamily.LINUX_ARM):
            logging.info("##### Fix RPATH in wheel's libraries #####")
            for whl in glob.iglob("{}/*.whl".format(wheel_path)):
                whl_name = os.path.relpath(whl, wheel_path)
                logging.info("Processing: %s", whl_name)
                if not wheel_fix_rpath(whl) and self.ctx[self.FailIfBrokenRpath.name]:
                    raise common.errors.TaskFailure(
                        "Can't fix RPATH in wheel {} libraries. See logs for more info.".format(whl_name)
                    )
            logging.info("#####")

        if not os.path.os.listdir(wheel_path):
            raise common.errors.TaskFailure(
                "Building wheel of {} package failed".format(self.ctx[self.PackageName.name])
            )
        name, version = self.get_package_version(wheel_name, wheel_path)
        logging.info("`%s:%s` package successfully converted to wheel format", name, version)
        prepared_resources = self.prepared_resources()
        if prepared_resources:
            id_ = prepared_resources[0][0]
            sdk_channel.channel.sandbox.set_resource_attribute(id_, "name", name)
            sdk_channel.channel.sandbox.set_resource_attribute(id_, "version", version)
            sdk_channel.channel.sandbox.set_resource_attribute(id_, "platform", platform.platform())
            sdk_channel.channel.sandbox.set_resource_attribute(id_, "ttl", "inf")
            self.save_parent_task_resource(wheel_path, id_)
            logging.info("Resource #%s saved to parent task", id_)
        else:
            res = self.create_resource(
                description="Python wheel of {pack} for {plat}".format(
                    pack=name,
                    plat=platform.platform()
                ),
                resource_path=wheel_path,
                resource_type=self.resource_type,
                attributes={
                    "name": name,
                    "version": version,
                    "platform": platform.platform(),
                },
                arch=self.client_info["arch"]
            )
            logging.info("Resource #%s with specified package successfully created", res.id)


__Task__ = BuildPyPackage
