# -*- coding: utf-8 -*-

import logging

import sandbox.projects.common.constants as consts

from sandbox.projects.YobjectBinaryBuild import YobjectBinaryBuild
from sandbox.projects.YobjectDataBuild import YobjectDataBuild

from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.network import is_port_free
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
import sandbox.projects.common.build.parameters as build_params

import diff_match_patch

import copy
import json
import requests
import re
import time

TASKS_GROUP_NAME = 'Build tasks with resources'
SVN_BRANCHES_GROUP_NAME = 'Svn branches to build'

BUILD_TARGETS_PARAM_NAME = 'targets'
EXECUTABLES_BUILD_TARGETS = 'dict/yobect/daemon'
DATA_BUILD_TARGETS = 'dict/yobject/data'

BRANCH_PARAM_NAME = 'checkout_arcadia_from_url'

DISK_SPACE_FOR_DATA_TASK = 120 * 1024  # 120 GB


class Ammo:
    def __init__(self):
        self.path = None
        self.method = None
        self.headers = dict()
        self.body = None

    def do_request(self, url, using_file=False):
        if self.method == 'POST':
            try:
                data = json.loads(self.body)
            except:
                data = self.body
            response = requests.post(url + self.path, json=data, headers=self.headers)
        else:
            response = requests.get(url + self.path, headers=self.headers)

        return response.text.encode('utf-8')

    def __str__(self):
        return '\tPATH: %s\n\tHEADERS: %s\n\tBODY: %s\n\n' % (str(self.path), str(self.headers), str(self.body))


def read_ammos_from_phantom(lines_iter):
    ammo = Ammo()
    ammo.body = ''
    content_len = 0
    unreaded_ammo_bytes = None
    headers_read = False
    for line in lines_iter:
        line = line

        if unreaded_ammo_bytes is None:
            if not line.strip():
                continue
            try:
                unreaded_ammo_bytes = int(line.strip().split(' ')[0])
            except:
                raise ValueError("Bad phantom format")
            continue

        # sys.sdterr.write('%s [%d, %d, %d]\n' % (line.strip(), len(line), len(line.strip()), unreaded_ammo_bytes)

        if line.startswith("GET") or line.startswith("POST"):
            parts = line.strip().split(' ')
            ammo.path = parts[1]
            ammo.method = parts[0]

        elif line.startswith("Content-Length:"):
            content_len = int(line.split(":")[1])
        elif not line.strip():
            headers_read = True
        elif not headers_read:
            header_key, header_value = line.strip().split(":", 1)
            ammo.headers[header_key] = header_value
        elif headers_read:
            ammo.body += line

        unreaded_ammo_bytes -= len(line)
        if unreaded_ammo_bytes < 0:
            raise ValueError("Bad phantom format")

        if not unreaded_ammo_bytes:
            if content_len and len(ammo.body) != content_len:
                raise ValueError("Bad phantom format")

            yield ammo

            ammo = Ammo()
            ammo.body = ''
            headers_read = False
            unreaded_ammo_bytes = None
            content_len = 0


def check_yobject_daemon_ready(host):
    try:
        response = requests.get('%s/ready' % host)
        return response.status_code == 200
    except:
        return False


def patch_cofig_for_testing(src_filename, dst_filename):
    config = open(src_filename, 'r').read()
    config = config.replace('entitysearch.yandex.net', 'entitysearch-test.yandex.net')

    # Long timeout on testing machines
    config = re.sub('timeout = [0-9]+', 'timeout = 60000', config)

    open(dst_filename, 'w').write(config)


def check_host_availability(config_path):
    config = open(config_path, 'r').read().replace('\n', '')

    for host_and_port_patter in re.finditer('host = "([^"]+)";[ \t]*port = ([0-9]+)', config):
        host = host_and_port_patter.group(1)
        port = int(host_and_port_patter.group(2))
        url = "http://%s:%d" % (host, port)

        # TODO: Request access from Sandbox to Maps and UGC source:
        IGNORED_HOSTS = ['http://geobase.qloud.yandex.ru', 'http://search.maps.yandex.net', 'http://mob.search.yandex.net']
        if any([url.startswith(ignored_host) for ignored_host in IGNORED_HOSTS]):
            continue

        try:
            requests.get(url, timeout=60)
            logging.debug('[TEST AVAILABILITY] Responce from {}'.format(url))
        except requests.exceptions.RequestException as e:
            logging.exception(str(e))
            raise SandboxTaskFailureError('[TEST AVAILABILITY] Bad GET request for "%s": %s' % (url, str(e)))


def filter_dump(src_filename, dst_filename):
    responce_readed = False
    with open(src_filename, 'r') as input_file:
        with open(dst_filename, 'w') as output_file:
            for line in input_file:
                if line.strip() == 'ACCESS_LOG':
                    responce_readed = True

                if responce_readed:
                    output_file.write(line)
                    continue
                result = None
                try:
                    result = json.loads(line)
                except:
                    continue
                if type(result) is dict:
                    # Remove unstable markup depends data from tests
                    result.pop('tags', None)

                pretty_json = json.dumps(result, sort_keys=True, ensure_ascii=False, indent=4, separators=(',', ': '))
                output_file.write(pretty_json.encode('utf-8') + '\n')


class ConfigPath(parameters.SandboxStringParameter):
    name = 'config_path'
    description = 'Config path'
    default_value = 'arcadia:/arc/trunk/arcadia/dict/yobject/daemon/config.cfg'


class CodeBranch(parameters.SandboxArcadiaUrlParameter):
    name = 'code_branch'
    description = 'Arcadia branch for code. Will not take effect if executables task will be specified'
    required = False
    group = SVN_BRANCHES_GROUP_NAME


class DataBranch(parameters.SandboxArcadiaUrlParameter):
    name = 'data_branch'
    description = 'Arcadia branch for data. Will not take effect if data task will be specified'
    required = False
    group = SVN_BRANCHES_GROUP_NAME


class LastReleaseTask(parameters.TaskSelector):
    name = 'last_release_task'
    description = 'Last release task'
    task_type = 'BUILD_YOBJECT'
    required = True
    do_not_copy = True


class AmmoPhantomResource(parameters.LastReleasedResource):
    name = 'ammo_phantom_resource'
    description = 'Resource with POST requests for testing'
    required = True
    resource_type = 'PHANTOM_REQUESTS'


class YobjectBinaryPatch(parameters.SandboxStringParameter):
    name = 'binary_patch'
    description = 'Patch for binary build'
    required = False
    multiline = True


class BuildYobject(task.SandboxTask):
    """
        Таск для сборки и последующего релиза Yobject
    """

    type = 'BUILD_YOBJECT'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = (
        ConfigPath, build_params.BuildType, CodeBranch, DataBranch, LastReleaseTask, AmmoPhantomResource, YobjectBinaryPatch
    )

    def on_release(self, additional_parameters):
        task.SandboxTask.on_release(self, additional_parameters)

    def create_executable_build_task(self):
        input_parameters = {
            build_params.BuildType.name: self.ctx[build_params.BuildType.name],
            BUILD_TARGETS_PARAM_NAME: EXECUTABLES_BUILD_TARGETS,
            BRANCH_PARAM_NAME: self.ctx[CodeBranch.name],
            consts.CHECK_RETURN_CODE: True,
            consts.ARCADIA_PATCH_KEY: self.ctx['binary_patch']
        }
        self.ctx[YobjectBinaryBuild.type] = self.create_subtask(
            task_type=YobjectBinaryBuild.type,
            description='Yobject executable build',
            arch='linux',
            input_parameters=input_parameters,
        ).id

    def create_data_build_task(self):
        input_parameters = {
            build_params.BuildType.name: self.ctx[build_params.BuildType.name],
            BUILD_TARGETS_PARAM_NAME: DATA_BUILD_TARGETS,
            BRANCH_PARAM_NAME: self.ctx[DataBranch.name],
            consts.CHECK_RETURN_CODE: True
        }
        self.ctx[YobjectDataBuild.type] = self.create_subtask(
            task_type=YobjectDataBuild.type,
            description='Yobject data build',
            arch='linux',
            input_parameters=input_parameters,
            execution_space=DISK_SPACE_FOR_DATA_TASK
        ).id

    def get_executable_build_task(self):
        if YobjectBinaryBuild.type not in self.ctx:
            self.create_executable_build_task()
        return self.ctx[YobjectBinaryBuild.type]

    def get_data_build_task(self):
        if YobjectDataBuild.type not in self.ctx:
            self.create_data_build_task()
        return self.ctx[YobjectDataBuild.type]

    def get_tasks(self):
        return self.get_executable_build_task(), self.get_data_build_task()

    def wait_children(self):
        tasks = self.get_tasks()

        need_wait = False
        for task_id in tasks:
            task_obj = channel.sandbox.get_task(task_id)
            if not task_obj:
                raise SandboxTaskFailureError('No build task with id {}'.format(task_id))

            is_done = task_obj.is_done()
            if not need_wait and not is_done:
                need_wait = True

            if is_done and not task_obj.is_ok():
                raise SandboxTaskFailureError('Build task {} is not in OK state'.format(task_id))

        if need_wait:
            self.wait_tasks(tasks=tasks, statuses=tuple(self.Status.Group.FINISH) + tuple(self.Status.Group.BREAK), wait_all=True)

    def create_dump_resource(self,
                             yobject_bin_resource,
                             yobject_config_resource,
                             yobject_data_resource,
                             daemon_port):
        YOBJECT_DAEMON_ACCESS_LOG = 'yobject.access_log.txt'
        YOBJECT_DAEMON_STDERR = 'yobject.sdterr.txt'
        YOBJECT_DAEMON_STDOUT = 'yobject.stdout.txt'

        IGNORED_LOG_FIELDS = ['request', 'user_agent', 'timestamp']
        RPS = 50

        yobject_bin_path = self.sync_resource(yobject_bin_resource)
        yobject_data_path = self.sync_resource(yobject_data_resource)
        config_resource_path = self.sync_resource(yobject_config_resource)

        yobject_config_path = 'config.%s.patched.cfg' % yobject_config_resource.id
        patch_cofig_for_testing(config_resource_path, yobject_config_path)
        check_host_availability(yobject_config_path)

        ammo_phantom_path = self.sync_resource(self.ctx['ammo_phantom_resource'])

        if not is_port_free(daemon_port):
            raise SandboxTaskFailureError("Daemon port %s is not free" % daemon_port)

        yobject_daemon_cmd = [yobject_bin_path,
                              yobject_config_path,
                              '-v', 'data_dir=%s' % yobject_data_path,
                              '-v', 'host_black_list=%s/%s' % (yobject_data_path, 'hostblacklist.txt'),
                              '-v', 'host_white_list=%s/%s' % (yobject_data_path, 'hostwhitelist.txt'),
                              '-v', 'object_black_list=%s/%s' % (yobject_data_path, 'objectblacklist.txt'),
                              '-v', 'accesslog=%s' % YOBJECT_DAEMON_ACCESS_LOG,
                              '-p', str(daemon_port)]
        proc = run_process(yobject_daemon_cmd,
                           log_prefix='daemon',
                           wait=False,
                           stderr=open(YOBJECT_DAEMON_STDERR, 'w'),
                           stdout=open(YOBJECT_DAEMON_STDOUT, 'w'))

        while not check_yobject_daemon_ready('http://localhost:%d' % daemon_port):
            time.sleep(5)

        dump_filename = 'yobject.dump.%s.%s.%s.%d.txt' % (yobject_bin_resource.id, yobject_config_resource.id, yobject_data_resource.id, daemon_port)
        with open(ammo_phantom_path, 'r') as ammo_file:
            with open(dump_filename, 'w') as responce_file:
                for iter, ammo in enumerate(read_ammos_from_phantom(ammo_file)):
                    if iter > 0 and (iter % RPS) == 0:
                        time.sleep(1)
                    responce = ammo.do_request('http://localhost:%d' % daemon_port, using_file=True)
                    responce_file.write(responce.strip() + '\n')

        proc.kill()

        with open(dump_filename, 'a') as result_dump:
            with open(YOBJECT_DAEMON_ACCESS_LOG, 'r') as access_log_file:
                result_dump.write('ACCESS_LOG\n')
                for line in access_log_file:
                    for field in line.split('\t'):
                        field = field.strip()
                        if field and not any([field.startswith(f + '=') for f in IGNORED_LOG_FIELDS]):
                            result_dump.write(field + '\n')

            for result_filename, block_name in ((YOBJECT_DAEMON_STDERR, 'STDERR'), (YOBJECT_DAEMON_STDOUT, 'STDOUT')):
                result_dump.write(block_name + '\n')
                with open(result_filename, 'r') as result_file:
                    for line in result_file:
                        if line.strip():
                            result_dump.write(line)

        filter_dump(dump_filename, dump_filename + '.filtered')

        dump_resource = self.create_resource(
            'YObject-daemon dump for binary(#%s), config(#%s) and data(#%s)' % (
                yobject_bin_resource.id,
                yobject_config_resource.id,
                yobject_data_resource.id
            ),
            dump_filename + '.filtered',
            resource_types.YOBJECT_DUMP
        )
        self.mark_resource_ready(dump_resource)
        return dump_resource

    """
        Resources functions
    """

    def get_yobject_dump_resource(self, build_yobject_task_id):
        for resource in apihelpers.list_task_resources(build_yobject_task_id):
            if resource.type == resource_types.YOBJECT_DUMP:
                return channel.sandbox.get_resource(resource.id)
        return None

    def create_config_resource(self):
        CONFIG_PATH = 'config.cfg'
        Arcadia.export(self.ctx['config_path'], CONFIG_PATH)
        created_resource = self.create_resource('Yobject config', CONFIG_PATH, resource_types.YOBJECT_CONFIG)
        self.mark_resource_ready(created_resource)
        return created_resource

    def grab_resources_from_task(self, task_id, res_types=None):
        if not res_types:
            res_types = []
        for resource in apihelpers.list_task_resources(task_id):
            if resource.type in res_types:
                yield self.grab_resource(resource)

    def grab_resource(self, resource_id):
        resource = channel.sandbox.get_resource(resource_id)
        src_path = self.sync_resource(resource.id)
        dst_path = resource.file_name
        copy_path(src_path, dst_path)

        attributes = copy.copy(resource.attributes)
        attributes.pop('released', None)
        created_resource = self.create_resource(resource.description, dst_path, resource.type,
                                                arch=resource.arch, attributes=attributes)
        self.mark_resource_ready(created_resource)
        return created_resource

    def create_resources(self):
        binary = self.grab_resources_from_task(self.ctx[YobjectBinaryBuild.type], [resource_types.YOBJECT_EXECUTABLE]).next()
        config = self.create_config_resource()
        data = self.grab_resources_from_task(self.ctx[YobjectDataBuild.type], [resource_types.YOBJECT_DATA]).next()

        return binary, config, data

    def create_resource_diff(self, old_resource_id, new_resource_id, description):
        old_resource = self.sync_resource(old_resource_id)
        new_resource = self.sync_resource(new_resource_id)
        old_content = open(old_resource, 'r').read()
        new_content = open(new_resource, 'r').read()

        dmp = diff_match_patch.diff_match_patch()
        diffs = dmp.diff_main(old_content[:10], new_content[:10])
        dmp.diff_cleanupSemantic(diffs)
        self.ctx['last_release_diff_html'] = dmp.diff_prettyHtml(diffs)

        with open('diff_with_last_release.html', 'w') as diff_file:
            diff_file.write(self.ctx['last_release_diff_html'])
        diff_resource = self.create_resource(description,
                                             'diff_with_last_release.html',
                                             resource_types.YOBJECT_DIFF)
        self.mark_resource_ready(diff_resource)
        return diff_resource

    def on_execute(self):
        self.wait_children()
        binary_resource, config_resource, data_resource = self.create_resources()

        old_dump_resource = self.get_yobject_dump_resource(self.ctx['last_release_task'])
        if old_dump_resource is None:
            raise SandboxTaskFailureError('Build task {} has not dump resource'.format(self.ctx['last_release_task']))

    @property
    def footer(self):
        if self.ctx.get('last_release_diff_html'):
            return self.ctx['last_release_diff_html']
        else:
            return ''


__Task__ = BuildYobject
