# coding: utf-8
import shutil
import os
import logging
import tarfile

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = ''
    required = True


class BuildServiceBundle(SandboxTask):
    type = 'BUILD_ZOOLOCK'
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [GitTagParameter]

    SERVICE_NAME = 'zoolock'
    GIT_URL = 'https://git.qe-infra.yandex-team.ru/scm/runtime/zoolock.git'

    CHECKOUT_PATH = SERVICE_NAME
    TGZ_PATH = '{}.tar.gz'.format(SERVICE_NAME)

    BUNDLE_RESOURCE = resource_types.ZOOLOCK_BUNDLE
    CONFIG_PRESTABLE_RESOURCE = resource_types.ZOOLOCK_CONFIG_PRESTABLE

    def _checkout(self):
        tag = self.ctx.get('tag')
        assert tag, 'Trying to fetch project from git, but no tag specified'
        run_process(
            ['git', 'clone', self.GIT_URL, self.CHECKOUT_PATH],
            log_prefix='git_clone',
            shell=True,
        )

    def _install_package(self, virtualenv_path):
        pip_path = os.path.join(virtualenv_path, 'bin', 'pip')
        # install service dependencies into virtualenv
        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '-r', 'requirements.txt'],
                    log_prefix='reqs_pip_install',
                    work_dir=self.CHECKOUT_PATH)
        # now install zoolock itself
        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '.'],
                    log_prefix='zoolock_pip_install',
                    work_dir=self.CHECKOUT_PATH)

    def _install_scripts_package(self, virtualenv_path):
        """
        Special magic to install zoolock package.
        """
        # copy library
        src_dir = os.path.join(self.CHECKOUT_PATH)
        dst_dir = os.path.join(virtualenv_path, 'zoolock')
        shutil.copytree(src_dir, dst_dir)

    def _make_resource(self, virtualenv_path):
        logging.info('creating tgz file')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(virtualenv_path):
                tar.add(os.path.join(virtualenv_path, entry), entry)
        self.create_resource(
            description='{} virtualenv tgz {}'.format(self.SERVICE_NAME, self.ctx['tag']),
            resource_path=self.TGZ_PATH,
            resource_type=self.BUNDLE_RESOURCE,
            arch='linux'
        )
        logging.info('releasing configs: prestable')
        self.create_resource(
            description='{} prestable config {}'.format(self.SERVICE_NAME, self.ctx['tag']),
            resource_path=os.path.join(virtualenv_path, 'zoolock', 'configs', 'prestable.json'),
            resource_type=self.CONFIG_PRESTABLE_RESOURCE,
            arch='linux'
        )

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('tag'), None

    def on_execute(self):
        """
        Plan is:
        * create virtualenv
        * git clone with specified tag
        * install service there
        * create scripts installation
        * copy config
        * create tgz with bundle
        """
        virtualenv_path = self.path('{}_build'.format(self.SERVICE_NAME))
        # create virtualenv using *skynet* python
        run_process(['/skynet/python/bin/virtualenv', '--no-site-packages', virtualenv_path])

        self._checkout()
        self._install_package(virtualenv_path)
        run_process(['/skynet/python/bin/virtualenv', '--relocatable', virtualenv_path])

        self._install_scripts_package(virtualenv_path)
        self._make_resource(virtualenv_path)


__Task__ = BuildServiceBundle
