# coding: utf-8
import shutil
import os
import logging
import tarfile
import time

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path
from sandbox.projects.common.utils import get_bsconfig
from sandbox.projects.common import apihelpers
from sandbox.projects.common.nanny import nanny


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = ''
    required = True


class BuildServiceBundle(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_ZOOREST'
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [GitTagParameter]

    SERVICE_NAME = 'zoorest'
    GIT_URL = 'https://git.qe-infra.yandex-team.ru/scm/search_infra/zoorest.git'
    CHECKOUT_PATH = SERVICE_NAME
    TGZ_PATH = '{}.tar.gz'.format(SERVICE_NAME)

    BUNDLE_RESOURCE = resource_types.ZOOREST_BUNDLE
    SHARD_RESOURCE = resource_types.ZOOREST_BUNDLE_SHARD

    def _get_iss_shards(self):
        iss_shards_resource = apihelpers.get_last_resource(resource_types.ISS_SHARDS)
        if not iss_shards_resource:
            raise Exception('There is no iss_shards resource')

        return self.sync_resource(iss_shards_resource)

    def _create_shard(self):
        shard_name = '{0}-{1}-{2}'.format(self.SERVICE_NAME, self.ctx['tag'], time.strftime("%Y%m%d_%H%M%S"))
        self.ctx['shard_name'] = shard_name
        shard_path = self.path(shard_name)
        remove_path(shard_path)
        os.mkdir(shard_name)
        shutil.copy(self.TGZ_PATH, shard_name)
        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_name, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(self.TGZ_PATH)
            )
        self.create_resource(
            description='{} {} shard'.format(self.SERVICE_NAME, self.ctx['tag']),
            resource_path=shard_path,
            resource_type=self.SHARD_RESOURCE,
            arch='linux'
        )
        # do initialize shard
        run_process(
            ['perl', get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="bs_config",
            work_dir=self.path()
        )

        # init in iss shards
        iss_shards = self._get_iss_shards()
        logging.info('iss_shards register')
        run_process(
            [
                iss_shards,
                'register',
                shard_name,
            ],
            log_prefix="iss_shards_register",
            work_dir=self.path(),
        )

        # in order to get same rbtorrent both for shard and resource we remove
        # some leftovers which are not shared by bsconfig
        for i in ['_lock.tmp', 'shard.state']:
            os.remove(os.path.join(shard_path, i))

    def _checkout(self):
        tag = self.ctx.get('tag')
        assert tag, 'Trying to fetch project from git, but no tag specified'
        run_process(
            ['git', 'clone', self.GIT_URL, self.CHECKOUT_PATH],
            log_prefix='git_clone',
            shell=True,
        )
        run_process(
            ['git', 'checkout', tag],
            work_dir=self.CHECKOUT_PATH,
            log_prefix='git_co',
            shell=True)

    def _install_package(self, virtualenv_path):
        pip_path = os.path.join(virtualenv_path, 'bin', 'pip')
        # install service dependencies into virtualenv
        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '-r', 'requirements.txt'],
                    log_prefix='reqs_pip_install',
                    work_dir=self.CHECKOUT_PATH)
        # now install zoorest itself
        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     '.'],
                    log_prefix='zoorest_pip_install',
                    work_dir=self.CHECKOUT_PATH)

    def _install_scripts_package(self, virtualenv_path):
        """
        Special magic to install zoorest package.
        """
        # copy library
        src_dir = os.path.join(self.CHECKOUT_PATH, 'src', 'zoorest')
        dst_dir = os.path.join(virtualenv_path, 'zoorest')
        shutil.copytree(src_dir, dst_dir)

    def _make_resource(self, virtualenv_path):
        logging.info('creating tgz file')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(virtualenv_path):
                tar.add(os.path.join(virtualenv_path, entry), entry)
        self.create_resource(
            description='{} virtualenv tgz {}'.format(self.SERVICE_NAME, self.ctx['tag']),
            resource_path=self.TGZ_PATH,
            resource_type=self.BUNDLE_RESOURCE,
            arch='linux'
        )

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('tag'), None

    def on_execute(self):
        """
        Plan is:
        * create virtualenv
        * git clone with specified tag
        * install service there
        * create scripts installation
        * copy config
        * create tgz with bundle
        * create shard
        """
        virtualenv_path = self.path('{}_build'.format(self.SERVICE_NAME))
        # create virtualenv using *skynet* python
        run_process(['/skynet/python/bin/virtualenv', '--no-site-packages', virtualenv_path])

        self._checkout()
        self._install_package(virtualenv_path)
        run_process(['/skynet/python/bin/virtualenv', '--relocatable', virtualenv_path])

        self._install_scripts_package(virtualenv_path)
        self._make_resource(virtualenv_path)
        self._create_shard()


__Task__ = BuildServiceBundle
