# -*- coding: utf-8 -*-
import os
import json
import logging

from .config import ZOOTOPIA_PROJECTS
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter, ResourceSelector
from sandbox.projects import resource_types
from sandbox.projects.common.build.YaMake import YaMakeTask, ya_make_build_params
from sandbox.projects.common.utils import check_if_tasks_are_ok
import sandbox.projects.common.build.parameters as build_params
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6


class ZootopiaProjectNameParameter(SandboxStringParameter):
    group = 'Project settings'
    name = 'project_name'
    required = True
    description = 'Project to build (name)'
    choices = ZOOTOPIA_PROJECTS.items()
    default_value = choices[0][1]


class ZootopiaProjectPathParameter(SandboxStringParameter):
    name = 'project_path'
    group = 'Project settings'
    required = True
    description = 'Project to build (path)'
    default_value = 'zootopia/hub/projectname'


class ZootopiaUsePreset(SandboxBoolParameter):
    name = 'zootopia_use_preset'
    description = 'Use preset projects'
    group = 'Project settings'
    default_value = True
    sub_fields = {
        'true': [ZootopiaProjectNameParameter.name],
        'false': [ZootopiaProjectPathParameter.name]
    }


class DockerTaskContainerParameter(ResourceSelector):
    name = 'docker_task_layer'
    description = 'Layer for docker build task'
    required = True
    resource_type = "LXC_CONTAINER"
    attrs = {'customization': 'docker'}
    default_value = '149265688'


class BuildZootopiaProject(YaMakeTask):
    """\
        Task for building zootopia project and wrapping it into Docker Image
    """
    type = 'BUILD_ZOOTOPIA_PROJECT'

    input_parameters = [ZootopiaUsePreset, ZootopiaProjectNameParameter,
                        ZootopiaProjectPathParameter, DockerTaskContainerParameter] \
        + build_params.get_arcadia_params() + ya_make_build_params()

    @property
    def project_path(self):
        param = ZootopiaProjectNameParameter
        if not self.ctx[ZootopiaUsePreset.name]:
            param = ZootopiaProjectPathParameter
        return self.ctx[param.name]

    def pre_build(self, source_dir):
        build_json_path = os.path.join(source_dir, self.project_path, 'build.json')
        self.ctx['build_json'] = json.load(open(build_json_path, 'r'))

    @property
    def current_project_config(self):
        return self.ctx['build_json']

    @property
    def current_project_resource_type(self):
        # return self.current_project_config.get('resource_type', resource_types.ARCADIA_PROJECT.name)
        return resource_types.ARCADIA_PROJECT.name

    def get_targets(self):
        return self.current_project_config.get('targets', [])

    def get_arts(self):
        return self.current_project_config.get('arts', [])

    def get_arts_source(self):
        return self.current_project_config.get('arts_source', [])

    def get_registry_tags(self):
        prefix = self.current_project_config.get('registry_tag', '') + ':'
        current_commit = self.ctx.get('ap_arcadia_revision')
        return [prefix + 'latest', '{}r{}'.format(prefix, current_commit)]

    def get_resources(self):
        return {'project': {'resource_type': self.current_project_resource_type}}

    def on_execute(self):
        if 'docker_task_id' not in self.ctx:
            YaMakeTask.on_execute(self)
            docker_task = self._create_task_BuildDockerImageV6()
            self.ctx['docker_task_id'] = docker_task.id
            self.wait_task_completed(task=self.ctx['docker_task_id'])
        else:
            check_if_tasks_are_ok(ids=[self.ctx['docker_task_id']])

    def _create_task_BuildDockerImageV6(self):
        sub_ctx = {
            BuildDockerImageV6.PackagedResource.name: self.ctx['ap_packs']['project'],
            BuildDockerImageV6.RegistryTags.name: self.get_registry_tags(),
            BuildDockerImageV6.ContainerParameter.name: self.ctx[DockerTaskContainerParameter.name],
        }
        logging.info('create {}({})'.format(BuildDockerImageV6.type, str(sub_ctx)))
        return self.create_subtask(
            task_type=BuildDockerImageV6.type,
            description='Build and publish Docker image for {}'.format(self.project_path),
            input_parameters=sub_ctx,
            arch='linux_ubuntu_14.04_trusty',
            execution_space=2 * 1024,
        )


__Task__ = BuildZootopiaProject
