# -*- coding: utf-8 -*-

import logging
import os

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common.build.parameters import ArcadiaPatch, ArcadiaUrl
import sandbox.projects.common.constants as consts
from sandbox.projects import resource_types

from lib import extract_info


COVERAGE_RESULTS = 'coverage_results'
CHILDREN = 'children'
REVISION = 'revision'
TARGET = 'target'
TARGETS = 'targets'
# style as in lcov report
FOOTER_CELL_STYLE_PREFIX = 'font-weight: bold; font-family: sans-serif; padding-left: 12px; padding-right: 4px; color:#000000; background-color:'
FOOTER_CELL_STYLE_BAD = FOOTER_CELL_STYLE_PREFIX + '#FF0000'
FOOTER_CELL_STYLE_MED = FOOTER_CELL_STYLE_PREFIX + '#FFEA20'
FOOTER_CELL_STYLE_GOOD = FOOTER_CELL_STYLE_PREFIX + '#A7FC9D'

MIN_MED_COVERAGE = 75.0
MIN_GOOD_COVERAGE = 90.0


def calc_nonempty_file_source_lines(filepath):
    ret = 0
    with open(filepath) as file_obj:
        for line in file_obj:
            line = line.strip()
            if line and not line.startswith('//'):
                ret += 1
    return ret


def is_source_file(filename):
    return not filename.endswith('_ut.cpp') and (filename.endswith('.h') or filename.endswith('.cpp'))


def calc_nonempty_directory_source_lines(directory):
    ret = 0
    for filename in os.listdir(directory):
        filepath = os.path.join(directory, filename)
        if os.path.isfile(filepath) and is_source_file(filename):
            ret += calc_nonempty_file_source_lines(filepath)
    return ret


class TargetLibsParameter(SandboxStringParameter):
    name = 'target_libs'
    description = 'Target libraries to test (separated by semicolon)'
    required = True


class CalcCoverage(SandboxTask):
    """
        Расчёт метрики покрытия кода модульными тестами.
        Поддерживается несколько целей. Расчёт по каждй из них
        производится по отдельности во избежание взаимовлияния
        тестов, чтобы иметь более точные измерения.
    """

    type = 'CALC_COVERAGE'

    input_parameters = (ArcadiaUrl, ArcadiaPatch, TargetLibsParameter)

    def initCtx(self):
        self.ctx['fail_on_any_error'] = True

    def report_extracting_coverage_info_error(self, target, exception):
        err = 'Error while calculating coverage for {}: {}'.format(target, str(exception))
        logging.error(err)
        self.set_info(err)

    def fix_revision(self):
        if not self.ctx.get(REVISION):
            arcadia_url = self.ctx[ArcadiaUrl.name]
            revision = Arcadia.get_revision(arcadia_url)
            self.ctx[REVISION] = revision
            self.ctx[ArcadiaUrl.name] = Arcadia.replace(arcadia_url, revision=revision)

    def extract_coverage_info_from_task(self, target, task_id):
        task = channel.sandbox.get_task(task_id)
        if not task:
            raise Exception('Child task {} was not found'.format(task_id))

        if not task.is_done():
            raise Exception('Child task ({}) is not done'.format(target))

        if not task.is_ok():
            raise Exception('Child task ({}) is not OK'.format(target))

        task_logs_resources = channel.sandbox.list_resources(resource_type=resource_types.TASK_LOGS, task_id=task_id)
        if not task_logs_resources:
            raise Exception('TASK_LOGS of ({}) were not found'.format(target))
        path = self.sync_resource(task_logs_resources[0].id)

        coverage_info = extract_info.extract_coverage_info_from_ya_make_output(open(path + '/ya_make.err.txt').read())
        additional_path = path + '/ya_make.err_1.txt'
        if os.path.exists(additional_path):
            coverage_info.update(extract_info.extract_coverage_info_from_ya_make_output(open(additional_path).read()))

        coverage_report_resource = channel.sandbox.list_resources(resource_type=resource_types.COVERAGE_REPORT, task_id=task_id)
        if not coverage_report_resource:
            raise Exception('COVERAGE_REPORT for ({}) was not found'.format(target))

        coverage_info['report'] = '{}/index.html'.format(coverage_report_resource[0].proxy_url)
        coverage_info[TARGET] = target

        return coverage_info

    def make_arcadia_url(self, target):
        url_components = self.ctx[ArcadiaUrl.name]
        rev_start = url_components.rfind('@')
        if rev_start != -1:
            url_components = url_components[:rev_start]
        url_components = url_components.strip('/')
        return Arcadia.replace(url_components + '/' + target, revision=self.ctx[REVISION])

    def has_unittest(self, target):
        return Arcadia.check(self.make_arcadia_url(target + '/ut'), revision=self.ctx[REVISION])

    def calc_library_lines_count(self, target):
        dest_directory = target.replace('/', '_')
        Arcadia.export(self.make_arcadia_url(target), dest_directory, revision=self.ctx[REVISION], depth='files')
        return calc_nonempty_directory_source_lines(dest_directory)

    def extract_coverage_info_manually(self, target):
        try:
            coverage_info = {
                'target': target,
                'lines_total': self.calc_library_lines_count(target),
                'lines_covered': 0,
                'lines_coverage': 0.0
            }
            self.ctx[COVERAGE_RESULTS][target] = coverage_info
        except Exception as e:
            self.report_extracting_coverage_info_error(target, e)

    def run_target_coverage_task(self, target):
        children = self.ctx.get(CHILDREN)
        if children and children.get(target):
            return

        if self.ctx[COVERAGE_RESULTS].get(target):
            return

        if not children:
            self.ctx[CHILDREN] = {}

        if not self.has_unittest(target):
            self.extract_coverage_info_manually(target)
            return

        input_parameters = {
            ArcadiaUrl.name: self.ctx[ArcadiaUrl.name],
            ArcadiaPatch.name: self.ctx[ArcadiaPatch.name],
            consts.BUILD_SYSTEM_KEY: consts.YMAKE_BUILD_SYSTEM,
            consts.BUILD_TYPE_KEY: consts.DEBUG_BUILD_TYPE,
            TARGETS: target + '/ut',
            consts.COVERAGE: True,
            consts.TESTS_REQUESTED: True,
            consts.COVERAGE_PREFIX_FILTER: target,
            consts.COVERAGE_EXCLUDE_REGEXP: '.*((\.pb\.(h|cc))|({}/.*/.*)|_ut.cpp)'.format(target),
            consts.DISABLE_TEST_TIMEOUT: True,
            consts.CLEAR_BUILD_KEY: False,
            'kill_timeout': 6 * 60 * 60
        }

        self.ctx[CHILDREN][target] = self.create_subtask(
            task_type='YA_MAKE',
            description='Coverage analysis for {} (revision {})'.format(target, self.ctx[REVISION]),
            input_parameters=input_parameters
        ).id

    def wait_children(self, tasks):
        logging.info('Wait children tasks: {}'.format(str(tasks)))
        need_wait = False
        for task_id in tasks:
            task_obj = channel.sandbox.get_task(task_id)
            if not task_obj:
                raise SandboxTaskFailureError('No task with id {}'.format(task_id))

            is_done = task_obj.is_done() or task_obj.is_stopped()
            if not need_wait and not is_done:
                need_wait = True

        if need_wait:
            self.wait_tasks(tasks=tasks, statuses=tuple(self.Status.Group.FINISH) + tuple(self.Status.Group.BREAK), wait_all=True)

    def on_execute(self):
        if COVERAGE_RESULTS not in self.ctx:
            self.ctx[COVERAGE_RESULTS] = {}

        self.fix_revision()

        for target in self.ctx[TargetLibsParameter.name].split(';'):
            self.run_target_coverage_task(target.strip())

        children_ids = [self.ctx[CHILDREN][target] for target in self.ctx.get(CHILDREN)]
        self.wait_children(children_ids)

        for target in self.ctx.get(CHILDREN):
            try:
                task_id = self.ctx[CHILDREN][target]
                logging.info('Extract coverage info from task {}'.format(task_id))
                result = self.extract_coverage_info_from_task(target, task_id)
                logging.info('Extracted coverage info from task {}'.format(task_id))
                self.ctx[COVERAGE_RESULTS][target] = result
            except Exception as e:
                self.report_extracting_coverage_info_error(target, e)

        extract_info.merge_coverage_info(self.ctx[COVERAGE_RESULTS])

    def generate_footer(self):
        results = self.ctx.get(COVERAGE_RESULTS)
        if not results:
            return

        sorted_targets = sorted(results.keys())

        directories = []
        lines = []
        functions = []
        branches = []

        for target in sorted_targets:
            result = results[target]
            report = result.get('report')
            if report:
                directories.append('<a href="{}">{}</a>'.format(report, target))
            else:
                directories.append(target)

            def format_coverage_result(prefix, result):
                if result.get(prefix + '_coverage') is None:
                    return '-'

                coverage = result.get(prefix + '_coverage')
                if coverage < MIN_MED_COVERAGE:
                    style = FOOTER_CELL_STYLE_BAD
                elif coverage < MIN_GOOD_COVERAGE:
                    style = FOOTER_CELL_STYLE_MED
                else:
                    style = FOOTER_CELL_STYLE_GOOD

                return '<div style="{}"><b>{} %</b> ({} / {})</div>'.format(style, coverage, result.get(prefix + '_covered'), result.get(prefix + '_total'))

            lines.append(format_coverage_result('lines', result))
            functions.append(format_coverage_result('functions', result))
            branches.append(format_coverage_result('branches', result))

        return {
            "<h4>Coverage results</h4>": {
                "header": [
                    {"key": "directory", "title": "Directory"},
                    {"key": "lines", "title": "Line coverage"},
                    {"key": "functions", "title": "Functions coverage"},
                    {"key": "branches", "title": "Branches coverage"}
                ],
                "body": {
                    "directory": directories,
                    "lines": lines,
                    "functions": functions,
                    "branches": branches
                }
            }
        }

    @property
    def footer(self):
        return self.generate_footer()


__Task__ = CalcCoverage
