# -*- coding: utf-8 -*-

import re


COVERAGE_REPORT_REGEXP_FORMAT_STRING = '\\s*{}\\.*:\\s*([0-9.]+)%\\s\\((\\d+) of (\\d+).*\\)'


def _try_match(line, regexp, prefix, ret):
    m = regexp.match(line)
    if m:
        ret[prefix + '_coverage'] = float(m.group(1))
        ret[prefix + '_covered'] = int(m.group(2))
        ret[prefix + '_total'] = int(m.group(3))
        return True

    return False


def extract_coverage_info_from_ya_make_output(output):
    found = False
    ret = {}
    lines_coverage_re = re.compile(COVERAGE_REPORT_REGEXP_FORMAT_STRING.format('lines'))
    functions_coverage_re = re.compile(COVERAGE_REPORT_REGEXP_FORMAT_STRING.format('functions'))
    branches_coverage_re = re.compile(COVERAGE_REPORT_REGEXP_FORMAT_STRING.format('branches'))
    matched = 0

    for line in output.split('\n'):
        if not found and line.find('Summary coverage rate:') != -1:
            found = True
        if found:
            if _try_match(line, lines_coverage_re, 'lines', ret) or \
                    _try_match(line, functions_coverage_re, 'functions', ret) or \
                    _try_match(line, branches_coverage_re, 'branches', ret):
                matched += 1
        if matched >= 3:
            break

    return ret


def _split_path(path):
    return [p for p in path.split('/') if p]


def _join_path_with_star(path):
    p = '/'.join(path)
    if p:
        p += '/'
    p += '*'
    return p


def _merge_coverage_info_field(dst, src, name):
    if name not in dst and name not in src:
        return

    value = dst.get(name, 0) + src.get(name, 0)
    dst[name] = value


def _merge_coverage_info(dst, src):
    _merge_coverage_info_field(dst, src, 'lines_total')
    _merge_coverage_info_field(dst, src, 'lines_covered')
    _merge_coverage_info_field(dst, src, 'functions_total')
    _merge_coverage_info_field(dst, src, 'functions_covered')
    _merge_coverage_info_field(dst, src, 'branches_total')
    _merge_coverage_info_field(dst, src, 'branches_covered')


def _calc_merged_field(info, name):
    total = name + '_total'
    covered = name + '_covered'
    if total not in info or covered not in info:
        return

    info[name + '_coverage'] = round(float(info[covered]) / float(info[total]) * 100.0, 1)


def _calc_merged_fields(info):
    _calc_merged_field(info, 'lines')
    _calc_merged_field(info, 'functions')
    _calc_merged_field(info, 'branches')


def merge_coverage_info(coverage_info):
    """
        Merge coverage info for subdirectories to such info for parent directory
    """
    targets = coverage_info.keys()
    prefixes_to_targets = {}
    for target in targets:
        splitted = _split_path(target)
        for i in xrange(len(splitted) + 1):
            fragment = _join_path_with_star(splitted[:i])
            if not prefixes_to_targets.get(fragment):
                prefixes_to_targets[fragment] = set()
            prefixes_to_targets[fragment].add(target)

    # make reverse dict to check duplicates
    target_sets_to_paths = {}
    for prefix, targets in prefixes_to_targets.iteritems():
        if len(targets) <= 1:
            continue
        const_targets = frozenset(targets)
        path = target_sets_to_paths.get(const_targets)
        if path is None or len(prefix) > len(path):
            target_sets_to_paths[const_targets] = prefix

    # merge coverage info
    for targets, path in target_sets_to_paths.iteritems():
        info = {'target': path}
        for target in targets:
            _merge_coverage_info(info, coverage_info[target])
        _calc_merged_fields(info)

        coverage_info[path] = info
