# -*- coding: utf-8 -*
import os
import logging
import shutil
import subprocess

from sandbox import sdk2
from sandbox.projects import resource_types

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import copy_path, get_logs_folder

from sandbox.projects.geosearch.tools.database_notifications import NotifyGeosearchDuty


class CalcMainRubric(NotifyGeosearchDuty, sdk2.Task):
    """
        Calculates machine rating for organizations
    """

    class Parameters(sdk2.task.Parameters):
        notify_parameters = NotifyGeosearchDuty.Parameters()
        rubric_calcer = sdk2.parameters.Resource('Main rubric calcer',
                                                 resource_type=resource_types.CALC_MAIN_RUBRIC_EXECUTABLE,
                                                 required=True)
        business_index = sdk2.parameters.Resource('Business index',
                                                  resource_type=resource_types.MAPS_DATABASE_BUSINESS)
        rubric_model = sdk2.parameters.Resource('Main rubric model',
                                                resource_type=resource_types.MAIN_RUBRIC_MODEL,
                                                required=True)
        dssm_model = sdk2.parameters.Resource('DSSM model',
                                              resource_type=resource_types.DSSM_MODEL,
                                              required=True)
        click_factors = sdk2.parameters.Resource('Click factors',
                                                 resource_type=resource_types.OTHER_RESOURCE,
                                                 required=True)
        index_shardmap = sdk2.parameters.Resource('Geobasesearch index shardmap',
                                                  resource_type=resource_types.ADDRS_BUSINESS_SHARDMAP)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 200 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def create_resource(self, descr, fname, res_type):
        rubric_resource = sdk2.Resource[res_type]
        current_rubric_resource = rubric_resource(self,
                                                  descr,
                                                  fname)
        rubric_data = sdk2.ResourceData(current_rubric_resource)
        rubric_data.ready()

    def on_execute(self):
        if self.Parameters.business_index:
            index = sdk2.ResourceData(self.Parameters.business_index).path

        calcer = sdk2.ResourceData(self.Parameters.rubric_calcer).path
        model = sdk2.ResourceData(self.Parameters.rubric_model).path
        dssm = sdk2.ResourceData(self.Parameters.dssm_model).path
        click_factors = sdk2.ResourceData(self.Parameters.click_factors).path

        shard_count = 0
        if self.Parameters.index_shardmap:
            index = './index'
            if os.path.exists(index):
                shutil.rmtree(index)
            index_task = self.Parameters.index_shardmap.task
            shard_tasks = index_task.Context.register_tasks
            shard_count = index_task.Parameters.shards_count
            for shard_id in xrange(shard_count):
                shard_resource = sdk2.Resource[resource_types.ADDRS_BUSINESS_SHARD].find(task=sdk2.Task[shard_tasks[shard_id]]).first()
                copy_path(str(sdk2.ResourceData(shard_resource).path),
                          os.path.join(index, str(shard_id)))

        cmd = ('{binary} '
               '{index}/ '
               '{click_factors} '
               '{dssm} '
               '--matrixnet-model {model}').format(binary=calcer,
                                                   index=index,
                                                   click_factors=click_factors,
                                                   dssm=dssm,
                                                   model=model)

        if shard_count > 0:
            cmd += (' --shard-count {shard_count}').format(shard_count=shard_count)

        logging.info('Running: %s' % cmd)

        log_file_path = get_logs_folder() + '/calc_main_rubric.out.txt'
        with open(log_file_path, 'w') as log_file:
            try:
                subprocess.Popen(cmd,
                                 shell=True,
                                 stdout=log_file,
                                 stderr=subprocess.STDOUT).wait()
            except subprocess.CalledProcessError as err:
                logging.info('%s command failed' % cmd)
                logging.info('Details %s:\n' % err)
                raise SandboxTaskFailureError('Calc main rubric failed')

        self.create_resource('main rubric bin', './main_rubrics.bin', resource_types.ORGS_MAIN_RUBRICS_BIN)
        self.create_resource('main rubric txt', './main_rubrics.txt', resource_types.ORGS_MAIN_RUBRICS_TXT)
