# encoding: utf-8

"""Calculate statistics about push delivery, send to statFace.

Uses binary resources created in: https://sandbox.yandex-team.ru/task/923088629/view (copy, check arcadia rev).
Don't forget to add a big ttl to the resource.
"""

import logging
import os

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import task_env
from sandbox.projects.common import time_utils as tu
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk import environments


class CalcPushMetrics(sdk2.Task):
    """Calculate push delivery stats and upload to statFace."""

    class Requirements(task_env.TinyRequirements):
        disk_space = 20 * 1024
        ram = 8 * 1024
        environments = [environments.PipEnvironment('python-step-client')]

    class Parameters(sdk2.Parameters):
        kill_timeout = 129600

        push_logs_parse_needed = sdk2.parameters.Bool(
            'Parse raw push logs.',
            default=True
        )
        metrika_logs_parse_needed = sdk2.parameters.Bool(
            'Parse metrika logs.',
            default=True
        )
        redir_logs_parse_needed = sdk2.parameters.Bool(
            'Parse redir logs.',
            default=True
        )
        logs_join_needed = sdk2.parameters.Bool(
            'Join raw logs, metrika logs and user_base logs.',
            default=True
        )
        metrics_calculation_needed = sdk2.parameters.Bool(
            'Calculate metrics.',
            default=True
        )
        publish_needed = sdk2.parameters.Bool(
            'Send metrics to statFace.',
            default=True
        )
        user_base_update_needed = sdk2.parameters.Bool(
            'Update device information base.',
            default=True
        )
        log_calc_days = sdk2.parameters.String(
            'Number of days to recalculate.',
            default='3'
        )
        log_date = sdk2.parameters.String(
            'Log date or comma-separated list of dates to calculate metrics from.',
            default=''
        )
        yt_pool = sdk2.parameters.String(
            'YT pool.',
            default='sup'
        )
        config_override = sdk2.parameters.String(
            'JSON-config. Overrides values in inner config. Use at your own risk.',
            default=''
        )
        send_push_final_ready = sdk2.parameters.Bool(
            'Send push_final_ready step event.',
            default=True
        )
        step_event_type = sdk2.parameters.String(
            'STEP sup_table_ready event type field',
            required=False
        )
        use_cloud = sdk2.parameters.Bool(
            'Use cloud pool',
            default=False
        )

    def on_execute(self):
        cmd_args = ['-f', 'home/search-functionality/sup/push_stats/', '-t', 'STATFACE_TOKEN']

        log_date = self.Parameters.log_date if self.Parameters.log_date else tu.date_ymd(delay_days=1)
        cmd_args.extend([
            '-d', log_date,
            '-l', self.Parameters.log_calc_days,
            '--yt_pool', self.Parameters.yt_pool,
            '--config_override', self.Parameters.config_override
        ])

        if self.Parameters.push_logs_parse_needed:
            logging.info('{time}:\tWill parse raw logs.'.format(time=tu.date_ymdhm()))
            cmd_args.append('-p')

        if self.Parameters.metrika_logs_parse_needed:
            logging.info('{time}:\tWill parse metrika logs.'.format(time=tu.date_ymdhm()))
            cmd_args.append('-m')

        if self.Parameters.redir_logs_parse_needed:
            logging.info('{time}:\tWill parse redir logs.'.format(time=tu.date_ymdhm()))
            cmd_args.append('--redir_logs_parse')

        if self.Parameters.logs_join_needed:
            logging.info('{time}:\tWill join Logs.'.format(time=tu.date_ymdhm()))
            cmd_args.append('-j')

        if self.Parameters.metrics_calculation_needed:
            logging.info('{time}:\tWill Calculate metrics.'.format(time=tu.date_ymdhm()))
            cmd_args.append('-s')

        if self.Parameters.publish_needed:
            logging.info('{time}:\tWill send metrics.'.format(time=tu.date_ymdhm()))
            cmd_args.append('-r')

        if self.Parameters.user_base_update_needed:
            logging.info('{time}:\tWill update user base.'.format(time=tu.date_ymdhm()))
            cmd_args.append('-u')

        if self.Parameters.use_cloud:
            logging.info('{time}:\tWill use cloud.'.format(time=tu.date_ymdhm()))
            cmd_args.append('-c')

        if self.Parameters.send_push_final_ready:
            logging.info('{time}:\tWill send push_final_ready step event.'.format(time=tu.date_ymdhm()))
        else:
            cmd_args.append('--dont-send-step-event')

        push_stats_binary_resources = sdk2.Resource.find(
            resource_type=resource_types.PUSH_STATS_TOOL,
            state='READY'
        ).limit(3)

        last_binary = sorted(list(push_stats_binary_resources), key=lambda x: x.created)[-1]
        last_binary_path = str(sdk2.ResourceData(last_binary).path)

        logging.info('Using PushStatsTool binary created on: ' + last_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', last_binary_path, cmd_args, '=======\n\n')

        os.environ['YQL_TOKEN'] = sdk2.Vault.data('robot_sup_yql_token')
        os.environ['STATFACE_TOKEN'] = sdk2.Vault.data('robot_sup_stat_token')
        os.environ['YT_TOKEN'] = sdk2.Vault.data('robot_sup_yt_token')
        os.environ['YT_PROXY'] = 'hahn'
        os.environ['REACTOR_TOKEN'] = sdk2.Vault.data('robot_sup_reactor_token')

        with sdk2.helpers.ProcessLog(self, logger="push_stats") as pl:
            sp.check_call([last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT)

        from step import Client
        if self.Parameters.step_event_type and self.Parameters.logs_join_needed:
            step_client = Client(host='https://step.sandbox.yandex-team.ru', timeout=(1, 1))
            event_id = step_client.create_event(
                name='sup_table_ready',
                params={'type': self.Parameters.step_event_type},
                sandbox_task_id=self.id)
            logging.info('STEP event: {}'.format(event_id))
