import datetime
import os

from sandbox import common
from sandbox import sdk2
import sandbox.sdk2.helpers

from sandbox.projects.geosearch.tools.database_notifications import NotifyGeosearchDuty


DATE_FORMAT = '%Y%m%d'


class CalcSimilarOrgsPreparationExecutable(sdk2.Resource):
    """ Executable that builds preparation tables on YT for the given range of days. """
    releasable = False
    any_arch = False
    executable = True
    auto_backup = True
    arcadia_build_path = 'extsearch/geo/tools/similar_orgs/create_preparat'


class DateParameter(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        if value:
            datetime.datetime.strptime(value, DATE_FORMAT)
        return value


class CalcSimilarOrgsPreparation(NotifyGeosearchDuty, sdk2.Task):
    """ Builds preparation tables on YT for future machine learning of similar org formulas. """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 512
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        notify_parameters = NotifyGeosearchDuty.Parameters()
        executable = sdk2.parameters.Resource('Executable to run', resource_type=CalcSimilarOrgsPreparationExecutable, required=True)

        with sdk2.parameters.Group('Range of dates to process') as dates_block:
            num_days = sdk2.parameters.Integer('Number of days', required=True, default=30)
            last_date = DateParameter('Last date (yesterday if not set)', required=False)
            rebuild = sdk2.parameters.Bool('Force rebuild', required=False, default=False)
            baobab = sdk2.parameters.Bool('Use baobab', required=False, default=False)

        with sdk2.parameters.Group('YT settings') as yt_block:
            output_path = sdk2.parameters.String('Output path', required=False)
            threads = sdk2.parameters.Integer('Threads', required=True, default=8)
            pool = sdk2.parameters.String('CPU Pool', required=False)

    def on_execute(self):
        exe = sdk2.ResourceData(self.Parameters.executable)

        args = [
            exe.path,
            '--threads', self.Parameters.threads
        ]
        if self.Parameters.output_path:
            args += ['--output', self.Parameters.output_path]
        if self.Parameters.rebuild:
            args += ['--rebuild']
        if self.Parameters.baobab:
            args += ['--baobab']

        if self.Parameters.last_date:
            last_date = datetime.datetime.strptime(self.Parameters.last_date, DATE_FORMAT).date()
        else:
            last_date = datetime.date.today() - datetime.timedelta(days=1)

        num_days = self.Parameters.num_days
        if num_days < 1:
            raise common.errors.TaskFailure('Invalid number of days: {}'.format(num_days))
        first_date = last_date - datetime.timedelta(days=num_days-1)

        args += [first_date.strftime(DATE_FORMAT), last_date.strftime(DATE_FORMAT)]

        env = os.environ.copy()
        env['YT_LOG_LEVEL'] = 'INFO'

        yt_token = sdk2.Vault.data(self.owner, 'yt-token')
        if yt_token:
            env['YT_TOKEN'] = yt_token

        if self.Parameters.pool:
            env['YT_POOL'] = self.Parameters.pool

        with sandbox.sdk2.helpers.ProcessLog(self, logger='create_preparat') as pl:
            sandbox.sdk2.helpers.subprocess.check_call(map(str, args), env=env, stdout=pl.stdout, stderr=pl.stderr)
