import os
import logging

import sandbox.sdk2 as sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.sandboxsdk.environments as environments
import sandbox.common.types.resource as ctr
import sandbox.common.errors as errors

from sandbox.projects.resource_types import MR_GEMINICL

from sandbox.projects.CalcZenEmbeddings.GrepBarNavigLog.yql_queries import grep_bar_navig_log, calc_clicks


class CalcZenEmbeddingsGrepBarNavigLog(sdk2.Task):
    """Grep one day of bar-navig-log"""

    class Parameters(sdk2.Task.Parameters):

        hosts_resource = sdk2.parameters.Resource(
            'Important hosts',
            required=True
        )

        hosts_to_path_resource = sdk2.parameters.Resource(
            'Hosts to path prefix resource',
            required=True
        )

        host_to_country_resource = sdk2.parameters.Resource(
            'Hosts to country resource',
            required=True
        )

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_vault_token = sdk2.parameters.String(
                'Your yt token name in vault',
                default='yt-token',
                required=True
            )
            yql_vault_token = sdk2.parameters.String(
                'Your yql token name in vault',
                default='YQL_TOKEN',
                required=True
            )

            input_table = sdk2.parameters.String(
                'Input table of bar-navig-log',
                required=True,
            )

            output_table = sdk2.parameters.String(
                'Result output table',
                required=True,
            )

            with sdk2.parameters.RadioGroup('Yt host') as yt_host:
                yt_host.values['hahn'] = yt_host.Value(value='Hahn', default=True)
                yt_host.values['banach'] = yt_host.Value(value='Banach')

        with sdk2.parameters.Group('Gemini parameters') as gemini_parameters:
            gemini_resource_id = sdk2.parameters.Resource(
                'MR Geminicl',
                resource_type=MR_GEMINICL,
                state=(ctr.State.READY,),
                required=True,
            )

            mr_gemini_user = sdk2.parameters.String(
                'MR gemini user',
                default='mr-any',
                required=True,
            )
            mr_gemini_job_count = sdk2.parameters.Integer(
                'MR gemini job count',
                default=100,
                required=True,
            )
            mr_gemini_max_rps = sdk2.parameters.Integer(
                'MR gemini max rps',
                default=15000,
                required=True,
            )

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        import yt.wrapper as yt
        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(
            db=self.Parameters.yt_host,
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token),
        )

        yt_proxy = '{}.yt.yandex.net'.format(self.Parameters.yt_host)

        yt.config['token'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
        yt.config['proxy']['url'] = yt_proxy

        gemini_runner = self.Parameters.gemini_resource_id

        if gemini_runner is None:
            raise errors.TaskError('No mr_geminicl founded')
        gemini_runner = sdk2.ResourceData(gemini_runner)

        input_table = str(self.Parameters.input_table)
        output_table = str(self.Parameters.output_table)

        temp_table = yt.create_temp_table()
        temp_table_unique = yt.create_temp_table()
        temp_table_canonized = yt.create_temp_table()

        with self.memoize_stage.first_yql_query(commit_on_entrance=False):
            request = yql_client.query(
                grep_bar_navig_log.format(
                    input_table=input_table,
                    temp_table=temp_table,
                    temp_table_unique=temp_table_unique,
                    hosts_resource=self.Parameters.hosts_resource.id,
                    hosts_to_path_resource=self.Parameters.hosts_to_path_resource.id,
                    host_to_country_resource=self.Parameters.host_to_country_resource.id,
                )
            )
            request.run()
            request.get_results(wait=True)

            if not request.is_success:
                raise errors.TaskError('first YQL query failed')

        logging.info('Start first gemini run')
        with sdk2.helpers.ProcessLog(self, logger='first mr_gemini') as pl:
            env = os.environ.copy()
            env['MR_RUNTIME'] = 'YT'
            env['MR_TMP'] = '//tmp/'
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)

            run = [
                str(gemini_runner.path),
                '--server', yt_proxy,
                '--user', str(self.Parameters.mr_gemini_user),
                '--input', temp_table_unique,
                '--output', temp_table_canonized,
                '--error-output', yt.create_temp_table(),
                '--job-count', str(self.Parameters.mr_gemini_job_count),
                '--max-rps', str(self.Parameters.mr_gemini_max_rps),
                '--type', 'weak',
            ]
            ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            if ret:
                raise errors.TaskError('first gemini run is failed')

        with self.memoize_stage.second_yql_query(commit_on_entrance=False):
            request = yql_client.query(
                calc_clicks.format(
                    output_table=output_table,
                    temp_table=temp_table,
                    temp_table_canonized=temp_table_canonized,
                )
            )
            request.run()
            request.get_results(wait=True)

            if not request.is_success:
                raise errors.TaskError('second YQL query failed')
