# -*- coding: utf-8 -*-
import logging
import os
import shutil
import tarfile
from contextlib import closing

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess
import sandbox.common.types.misc as ctm
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.common.build.tasks import YaMakeTemplate

from sandbox.projects.CeleryDashboard.resource_types import CELERY_DASHBOARD_API, CELERY_DASHBOARD_FRONT


npm_logger = logging.getLogger('npm_builder')


class BuildCeleryDashboard(release_integration.ReleaseToNannyAndYaDeployTask2, YaMakeTemplate.YaMakeTemplate):
    class Parameters(YaMakeTemplate.get_project_params([
        CELERY_DASHBOARD_API
    ])):
        pass

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('10.20.1'),
            environments.GCCEnvironment(version="5.3.0"),
        ]

    YP_TOKEN_YAV_SECRET_ID = "sec-01fvacjjs4vapyph262wba7qm2"

    def on_execute(self):
        with arcadiasdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url) as arcadia_path:
            arcadia_path = sdk2.Path(arcadia_path)
            self.make_dashboard_front_resource(arcadia_path)

        super(BuildCeleryDashboard, self).on_execute()

    def make_dashboard_front_resource(self, arcadia_path):
        self._make_front_resource(
            arcadia_path,
            resource=CELERY_DASHBOARD_FRONT,
            resource_description="Dashboard front build",
            resource_path='dashboard_builds.tar',
            work_dir='dashboard_front',
            build_path='build',
            new_build_path='dashboard_builds',
        )

    def _make_front_resource(
        self,
        arcadia_path,
        resource,
        resource_description,
        resource_path,
        work_dir,
        build_path,
        new_build_path,
    ):
        resource = resource(self, resource_description, resource_path)
        front_dir = str(arcadia_path / resource.arcadia_build_path)
        shutil.copytree(front_dir, work_dir)

        with sdk2.helpers.ProcessLog(self, logger=npm_logger) as pl:
            subprocess.check_call(
                ['npm', 'install', '--production', '--registry=https://npm.yandex-team.ru'],
                cwd=work_dir,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            subprocess.check_call(
                ['npm', 'run', 'build'],
                cwd=work_dir,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

        build_path = os.path.join(work_dir, build_path)
        shutil.move(build_path, new_build_path)

        resource_data = sdk2.ResourceData(resource)

        with closing(tarfile.open(str(resource_data.path), 'w')) as tar:
            tar.add(new_build_path)

        resource_data.ready()
