import sys
import json
import itertools
import subprocess
import collections

from sandbox import sdk2

from sandbox.common import errors


_DEPENDENCY_MASKS = {
    'search/begemot/apphost': [
        (True, 'search/begemot/apphost'),
        (True, 'search/begemot/core'),
        (True, 'search/begemot/status'),  # begemot core requires this
        (True, 'search/begemot/rules/grunwald/proto'),
        (True, 'search/begemot/rules/qtree/serialization/proto'),  # for wizard-compatible json
        (True, 'search/wizard/common'),  # XXX: ?
        (False, 'search/begemot'),
    ],

    'search/begemot/core': [
        (True, 'search/begemot/core/proto'),
        (True, 'search/begemot/status'),  # rule_set requires statistics
        (False, 'search/begemot'),
    ],

    'search/begemot/data': [
        (True, 'search/wizard/data/wizard'),
        (False, 'search/wizard'),
        (False, 'search/begemot'),
        (True, 'search/begemot/core/proto'),
        (True, 'search/begemot/rules/market/proto'),
        (True, 'search/begemot/rules/'),
        # thesaurus.gztproto
        (True, 'search/wizard/rules/proto'),  # TODO remove this one
    ],

    'search/begemot': [
        (True, 'search/wizard/common'),
        # Temporary hack because we cannot easily move gztprotos
        # Here we depend on music_magic.gztproto
        (True, 'search/wizard/rules/proto'),
        # Where do we actually want this folder to reside?
        # Wares rule uses it (and other sources)
        (True, 'search/wizard/entitysearch'),
        (False, 'search/wizard'),
    ],

    'search/wizard/common': [
        (True, 'search/begemot'),
        (True, 'search/wizard/common'),
        (False, 'search/wizard'),
    ],
}


_IGNORED_EDGES = {
    'search/begemot/rules$': ['search/begemot/rules'],
    'search/begemot/rules/thesaurus': ['search/begemot/rules/thesaurus/dynamic_rules'],  # pseudo-loop on includes
}


def _matches(path, mask):
    if mask.endswith('$'):
        return path == mask[:-1]
    return path == mask or path.startswith(mask + '/')


def _arc_graph(arcadia, targets):
    p = [str(arcadia / 'search' / t) for t in targets]
    g = subprocess.check_output([sys.executable, str(arcadia / 'ya'), 'dump', 'dir-graph', '--split'] + p)
    return json.loads(g)


def _arc_graph_bfs(graph, targets, kind):
    parents = targets.copy() if isinstance(targets, dict) else {target: set() for target in targets}
    queue = collections.deque(targets)
    while queue:
        target = queue.pop()
        for path in graph.get(target, {}).get(kind, []):
            for mask, ignored in _IGNORED_EDGES.items():
                if _matches(path, mask) and any(_matches(target, submask) for submask in ignored):
                    break
            else:
                if path not in parents:
                    queue.append(path)
                parents.setdefault(path, set()).add((target, kind))
    return parents


def _arc_graph_paths(deps, targets, roots=()):
    yield 'digraph {\n'
    ids = {}
    for i, path in enumerate(deps):
        ids[path] = i
    visited = set(targets)
    queue = collections.deque(targets)
    while queue:
        target = queue.pop()
        for path, kind in deps.get(target, []):
            style = "solid" if kind == "INCLUDE" else "dashed"
            yield '  {} -> {} [style="{}"]\n'.format(ids[path], ids[target], style)
            if path not in visited:
                visited.add(path)
                queue.append(path)
    for path in visited:
        color = "red" if path in targets else "green" if path in roots else "black"
        yield '  {} [label="{}", color="{}"]\n'.format(ids[path], path, color)
    yield '}\n'


def _arc_dep_check(graph, masks, out=None):
    masks = sorted(masks.items(), reverse=True)
    for path in sorted(graph):
        checks = [c for mask, c in masks if _matches(path, mask)]
        if not checks:
            continue
        # XXX: follow RECURSE first?
        deps = _arc_graph_bfs(graph, [path], "INCLUDE")
        disallowed = set()
        if deps[path]:
            disallowed.add(path)  # circular dependency
        for dir in deps:
            if dir == path:
                continue
            for allow, target in itertools.chain.from_iterable(checks):
                if dir != target and not dir.startswith(target + '/'):
                    continue
                if allow:
                    break
                disallowed.add(dir)
        if disallowed:
            dot = ''.join(_arc_graph_paths(deps, disallowed, [path])).encode('utf-8')
            name = path.replace('/', '_') + '.png'
            with open(name if out is None else str(out / name), 'w') as fd:
                proc = subprocess.Popen(['dot', '-Tpng'], stdin=subprocess.PIPE, stdout=fd)
                proc.communicate(dot)
            yield path, disallowed, name


class BegemotDependencyPathGraphs(sdk2.Resource):
    """A directory with graphs that show paths from a target (green) to some dependencies (red)"""


class CheckBegemotDependencies(sdk2.Task):
    """Fail if a target in search/wizard or search/begemot depends on a forbidden path"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 2 * 1024

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.String('Svn url for arcadia')
        checkout_arcadia_from_url.default_value = sdk2.svn.Arcadia.trunk_url()

    def on_execute(self):
        self.Context.graphs = {}
        self.Context.disallowed = {}
        url = self.Parameters.checkout_arcadia_from_url
        Arcadia = sdk2.svn.Arcadia

        svn_info = Arcadia.parse_url(url)
        arc = Arcadia.checkout(url, str(sdk2.path.Path().resolve() / 'arc'), depth='immediates')
        for p in ['search/wizard', 'search/begemot', 'devtools', 'build']:
            Arcadia.update(arc + '/' + p, set_depth='infinity', revision=svn_info.revision, parents=True)

        out = sdk2.path.Path().resolve() / 'out'
        out.mkdir(mode=0o755, parents=True, exist_ok=True)
        graph = _arc_graph(sdk2.path.Path(arc), ['wizard', 'begemot'])
        for path, disallowed, graph in _arc_dep_check(graph, _DEPENDENCY_MASKS, out):
            self.Context.graphs[path] = graph
            self.Context.disallowed[path] = list(disallowed)
        if not self.Context.graphs:
            open(str(out / '.empty'), 'w').close()
        res = BegemotDependencyPathGraphs(self, '', str(out))
        self.Context.graphs_root = res.id
        sdk2.ResourceData(res).ready()
        if self.Context.graphs:
            raise errors.TaskFailure('some targets depend on forbidden paths; see footer for details')

    @property
    def footer(self):
        if not self.Context.graphs:
            return
        rs = sdk2.Resource[self.Context.graphs_root]

        def render():
            yield '<table class="data_custom_fields t t_max t_cross">'
            for path, out in sorted(self.Context.graphs.items()):
                bad = self.Context.disallowed[path]
                yield '<tr>'
                yield '<td style="font-weight:bold">{}</td>'.format(path)
                yield '<td><a href="{}/{}">{}</a></td>'.format(rs.http_proxy, out, bad[0] if len(bad) == 1 else bad[0] + ' and more...')
                yield '</tr>'
            yield '</table>'
        return [{'content': ''.join(render())}]
