# -*- coding: utf-8 -*-
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk import environments

from sandbox.projects.common import apihelpers
from sandbox.projects import resource_types

from datetime import datetime
from time import mktime

import logging
import sys
import random
import re
import subprocess


class RazladkiInput(SandboxStringParameter):
    name = 'razladki_input'
    description = 'Razladki dict input (not str)'


class StTicket(SandboxStringParameter):
    name = 'ticket'
    description = 'ST ticket'


def parse_title(title):
    sp = title.split(":", 3)
    prefix = "fast" if "fast" in sp[0] else "daily"
    log = sp[1].split(" ")[-1].strip(" ")
    error = sp[2].strip(" ")
    return prefix, log, error


class CheckRazladkaParseLib(SandboxTask):
    type = 'CHECK_RAZLADKA_PARSELIB'

    input_parameters = [RazladkiInput, StTicket]
    environment = [environments.PipEnvironment("startrek_client", use_wheel=True)]

    def handle_failure(self, st_client, comment):
        logging.info("Error: %s " % comment)
        self.comment_st_ticket(comment, self.ticket, st_client)
        raise ValueError(comment)

    def __init__(self, *args, **kwargs):
        SandboxTask.__init__(self, *args, **kwargs)

    def get_grep_command(self):
        resource = apihelpers.get_last_resource(resource_types.MR_GREP)
        if not resource:
            return None
        return self.sync_resource(resource)

    def get_mapreduce(self):
        resource = apihelpers.get_last_resource(resource_types.MAPREDUCE_EXECUTABLE)
        if not resource:
            return None
        return self.sync_resource(resource)

    def get_data(self):
        return self.ctx['razladki_input']['markup'], self.ctx['razladki_input']['command']

    def get_date(self):
        return datetime.strptime(self.ctx['razladki_input']['markup']['date'], '%d.%m.%Y')

    def comment_st_ticket(self, comment, ticket, client, attachment=None):
        try:
            issue = client.issues[ticket]
            if attachment:
                issue.comments.create(text=comment, attachments=[attachment])
            else:
                issue.comments.create(text=comment)
        except Exception:
            raise ValueError(u'Can\'t add comment to ticket'), None, sys.exc_info()[2]

    def get_table_from_type(self, prefix, log):
        date = self.get_date()
        if prefix == 'fast':
            ts = mktime(date.timetuple())
            ts_cutted = int(ts - ts % 1800)
            table = 'fast_logs/user_sessions/%d/errors' % ts_cutted
        else:
            if 'spy_log' in log:
                table = 'spy_log/{}/user_sessions/raw/errors'.format(date.strftime('%Y%m%d'))
            elif 'market' in log:
                table = 'spy_log/{}/user_sessions/raw/errors'.format(date.strftime('%Y%m%d'))
            elif 'watch_log' in log:
                table = 'watch_log/{}/user_sessions/raw/errors'.format(date.strftime('%Y%m%d'))
            else:
                table = 'user_sessions/{}/errors'.format(date.strftime('%Y%m%d'))

        return table

    def on_execute(self):
        import startrek_client

        markup, command = self.get_data()
        logging.info('Markup: %s' % repr(markup))
        logging.info('Command: %s' % repr(command))

        # prepare startrack client
        st_client = startrek_client.Startrek(token=command['auth']['token'],
                useragent=command['auth']['user'], base_url=command['ticket_spec']['st_api_url'])
        self.ticket = markup['ticket_id']

        # sandbox url
        cutted_url = command["sandbox_spec"]["sandbox_url"].rstrip("/").rstrip("sandbox").rstrip("/")
        sandbox_task_final_url = "{}/task/{}/view".format(cutted_url, self.id)
        comment_url = "Sandbox task final url: " + sandbox_task_final_url + '\n----\n'

        # parse callback context
        if not markup.get('titles'):
            self.handle_failure(st_client, comment_url + 'No titles given to run callback')

        main_title = markup['titles'][0]
        logging.info('Look to %s param' % main_title)
        try:
            prefix, log, error = parse_title(main_title)
        except:
            self.handle_failure(st_client, comment_url + 'Failed to parse title %s\n' % main_title)

        if error == "Unknown error":
            self.handle_failure(st_client, comment_url + 'Don\'t know how to  handle "Unknown error"\n')

        if not markup.get('params'):
            self.handle_failure(st_client, comment_url + 'No params given to run callback\n')

        param = markup['params'][0]
        logging.info('Look to %s param' % param)

        server = command['mapreduce']['server']
        mr_user = command['mapreduce']['mr_user']
        table_dir = command['mapreduce']['table_path']

        if not markup.get('ticket_id'):
            self.handle_failure(st_client, comment_url + 'No ticket id given to run callback')

        table = self.get_table_from_type(prefix, log)
        # check if table exist
        mapreduce_binary = self.get_mapreduce()
        if not mapreduce_binary:
            self.handle_failure(st_client, comment_url + 'Haven\'t found mapreduce binary\n')

        list_command = 'MR_USER={} {} -server {} -list -exact {}'.format(mr_user, mapreduce_binary, server, table)
        logging.info('List command: %s' % list_command)
        try:
            answer = run_process(list_command, shell=True, stdout=subprocess.PIPE, wait=True, log_prefix='list').communicate()
            logging.info('List answer: %s' % repr(answer))
            if not answer or not answer[0]:
                self.handle_failure(st_client, comment_url + 'Table doesn\'t not exist: {}\n'.format(table))
        except SandboxSubprocessError, e:
            self.handle_failure(st_client, comment_url + 'MAPREDUCE list failed: {}'.format(e))

        # grep table
        ABC = "qwertyuiopasdfghjklzxcvbnm"
        result_table = table_dir + '/callback_' + log + "_" + self.get_date().strftime('%Y%m%d') + "".join(random.choice(ABC) for _ in xrange(4))

        grep_binary = self.get_grep_command()
        if not grep_binary:
            self.handle_failure(st_client, 'Haven\'t found grep binary\n')
        logging.info('MR_GREP binary: %s' % grep_binary)

        if error == 'Out error':
            regexp = '^(?!.*(createlib|parse_lib).*)'
        else:
            regexp = re.escape(error)

        grep_command = 'MR_USER={} {} -s {} -sub -k "^{}$" -sk "{}" -and -d  {} {}'.format(mr_user, grep_binary, server, log, regexp, result_table, table)
        logging.info('Grep command: %s' % grep_command)

        try:
            run_process(grep_command, shell=True, log_prefix='grep')
        except SandboxSubprocessError, e:
            self.handle_failure(st_client, comment_url + 'MR_GREP failed: {}'.format(e))

        # read grep results for comment
        read_file = 'record.txt'
        read_command = "MR_USER={} {} -server {} -read {} -subkey -count 2 > {}".format(mr_user, mapreduce_binary, server, result_table, read_file)
        logging.info('Read command: %s' % read_command)

        try:
            run_process(read_command, shell=True, log_prefix='read')
        except SandboxSubprocessError, e:
            self.handle_failure(st_client, comment_url + 'Read failed: {}'.format(e))

        with open(read_file) as cr:
            grep_result = [i.decode('utf-8', 'replace').strip("\n") for i in cr if len(i)]

        if not grep_result:
            self.handle_failure(st_client, comment_url + 'Empty grep result from command {}\nGrep_command: {}'.format(read_command, grep_command))

        comment_table = 'Table: {}\n----\n'.format(result_table)
        comment_grep_command = 'Grep command: {}\n----\n'.format(grep_command)
        comment_grep_result = '%%' + '\n'.join(grep_result[:2]) + '%%'
        comment = comment_url + 'Callback results:\n' + comment_table + comment_grep_command + comment_grep_result

        # read grep results for attach file
        attach_file = "100_records.txt"
        attach_command = "MR_USER={} {} -server {} -read {} -subkey -count 100 > {}".format(mr_user, mapreduce_binary, server, result_table, attach_file)
        logging.info('attach command: %s' % attach_command)

        try:
            run_process(attach_command, shell=True, log_prefix='attach')
        except SandboxSubprocessError, e:
            self.handle_failure(st_client, comment_url + 'Attach read failed: {}'.format(e))

        # send result to startrack
        try:
            self.comment_st_ticket(comment, self.ticket, st_client, attach_file)
        except ValueError:
            pass


__Task__ = CheckRazladkaParseLib
