# -*- coding: utf-8 -*-

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects import resource_types
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.TestDynamicModelsFeatures import TestDynamicModelsFeatures


class ModelsArchiveParameter(sp.ResourceSelector):
    name = 'models_archive_resource_id'
    description = 'Models archive'
    default_value = None
    required = True


class RelevFmlUnusedParameter(sp.ResourceSelector):
    name = 'relev_fml_unused_resource_id'
    description = 'relev_fml_unused executable'
    resource_type = 'RELEV_FML_UNUSED_EXECUTABLE'
    required = True


class FailOnBadFactorsParameter(sp.SandboxBoolParameter):
    name = 'fail_on_bad_factors'
    description = 'Fail if bad factors found'
    default_value = False


class CheckUnusedFactors(TestDynamicModelsFeatures):
    """
        * Показывает TG_REMOVED факторы, которые используются в какой-либо модели
        * Показывает TG_DEPRECATED факторы, которые не используются ни в одной из моделей
    """

    type = 'CHECK_UNUSED_FACTORS'

    input_parameters = [
        ModelsArchiveParameter,
        RelevFmlUnusedParameter,
        FailOnBadFactorsParameter,
        build_params.ArcadiaUrl,
    ]

    execution_space = 40000  # 40 Gb
    cores = 17

    @property
    def footer(self):
        bad_output = self.ctx.get('bad_output')
        if bad_output:
            return '<b>relev_fml_unused output:</b>' + '<br/>'.join(bad_output)
        else:
            return '<b>There are no bad factors</>'

    def on_execute(self):
        self.dofail = utils.get_or_default(self.ctx, FailOnBadFactorsParameter)
        self.arcadia_url = utils.get_or_default(self.ctx, build_params.ArcadiaUrl)
        self._run_check()

    def _run_check(self):
        relev_fml_unused = self.sync_resource(self.ctx[RelevFmlUnusedParameter.name])
        code_dir = Arcadia.get_arcadia_src_dir(self.arcadia_url)
        command = [
            relev_fml_unused,
            '--arc', code_dir,
            '--removed',
            '--unused'
        ]

        models_archive = self.sync_resource(self.ctx[ModelsArchiveParameter.name])
        command.extend(['--archive', models_archive])

        output = self.create_resource(
            'relev_fml_unused output',
            self.abs_path('relev_fml_unused_output'),
            resource_types.OTHER_RESOURCE,
        )
        with open(output.path, 'w') as file:
            p = process.run_process(command, stdout=file, stderr=file, check=False, wait=True)

        if p.returncode != 0:
            has_bad_factors = False
            output_lines = fu.read_lines(output.path)
            for line in output_lines:
                if line.startswith('web TG_REMOVED') or line.startswith('web TG_DEPRECATED'):
                    has_bad_factors = True
                    break

            if has_bad_factors:
                self.ctx['bad_output'] = output_lines
                if self.dofail:
                    eh.check_failed('Bad factors found')
            else:
                self.set_info('relev_fml_unused output:\n{}'.format('\n'.join(output_lines)))
                eh.check_failed('Error during relev_fml_unused execution')


__Task__ = CheckUnusedFactors
