# -*- coding: utf-8 -*-

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.environments as sdk_environments
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.common.types.client as ctc


class YtProxy(sdk_parameters.SandboxStringParameter):
    name = 'yt_proxy'
    description = 'YT proxy'
    required = True
    default = 'hahn'


class YtToken(sdk_parameters.SandboxStringParameter):
    name = 'yt_token'
    description = 'YT_TOKEN vault name'


class PathContains(sdk_parameters.SandboxStringParameter):
    name = 'path_contains'
    description = 'Delete only tables which path contains this'


class TablesTTL(sdk_parameters.DictRepeater, sdk_parameters.SandboxStringParameter):
    name = 'tables_ttl'
    description = 'Root path and ttl in mitunes'
    required = True


class CleanupYtDirectory(sdk_task.SandboxTask):
    """Remove old tables in YT"""

    type = "CLEANUP_YT_DIRECTORY"
    environment = [sdk_environments.PipEnvironment('yandex-yt'), ]
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = [
        YtProxy,
        YtToken,
        PathContains,
        TablesTTL,
    ]

    def table_is_too_old(self, mod_time, ttl):
        from datetime import datetime
        import time
        from dateutil import tz

        ts = time.strptime(mod_time, '%Y-%m-%dT%H:%M:%S.%fZ')
        d = datetime(*ts[0:6], tzinfo=tz.tzutc())
        return d < datetime.now(tz.tzutc()) - ttl

    def old_tables(self, yt, dirs_ttl, path_contains):
        path_filter = (lambda path: path_contains in path) if path_contains else None
        for dirpath, ttl in dirs_ttl.iteritems():
            for node in yt.search(
                dirpath,
                node_type="table",
                attributes=["modification_time"],
                path_filter=path_filter,
            ):
                if self.table_is_too_old(node.attributes["modification_time"], ttl):
                    yield node

    def on_execute(self):
        from yt.wrapper import YtClient
        from datetime import timedelta

        yt = YtClient(proxy=self.ctx[YtProxy.name])
        yt_token = self.ctx.get(YtToken.name)
        if yt_token:
            yt.config['token'] = self.get_vault_data(yt_token)

        dirs_ttl = {
            root: timedelta(minutes=int(ttl_minutes))
            for root, ttl_minutes in self.ctx[TablesTTL.name].iteritems()
        }

        for node in self.old_tables(yt, dirs_ttl, path_contains=self.ctx[PathContains.name]):
            yt.remove(node)


__Task__ = CleanupYtDirectory
