# coding: utf-8

import logging
import time

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk import errors
from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.sandboxapi import RELEASE_STATUSES
from sandbox.projects.porto import common as porto_common


class ParentLayer(ResourceSelector):
    name = 'parent_layer'
    description = 'Parent Layer'
    resource_type = porto_common.resource_types_with_prefixes("PORTO_LAYER_COCAINE")
    required = False


class DockerImageName(SandboxStringParameter):
    name = 'docker_image_name'
    description = 'Docker image name'
    required = True


class RegistryFQDN(SandboxStringParameter):
    name = 'registry_fqdn'
    description = 'Docker registry FQDN'
    default_value = 'registry.ape.yandex.net'
    required = True


class DockerfileCmd(SandboxStringParameter):
    name = 'dockerfile_cmd'
    description = 'Dockerfile CMD'
    required = True
    multiline = False


class CocaineLayersExportToRegistry(SandboxTask):
    type = 'COCAINE_LAYERS_EXPORT_TO_REGISTRY'
    client_tags = Tag.COCAINE & ctc.Tag.Group.LINUX
    cores = 17

    class VaultItemOwner(SandboxStringParameter):
        name = 'vault_item_owner'
        description = 'Vault item owner (task author by default)'
        group = 'Docker registry authentication'
        required = False

    class VaultItemName(SandboxStringParameter):
        name = 'vault_item_name'
        description = 'Vault item name (registry login)'
        group = 'Docker registry authentication'
        required = False

    class CheckVault(SandboxBoolParameter):
        name = 'check_vault'
        description = 'Authenticate in registry'
        group = 'Docker registry authentication'
        required = False
    CheckVault.sub_fields = {'true': [VaultItemName.name, VaultItemOwner.name]}

    input_parameters = [ParentLayer,
                         DockerImageName,
                         DockerfileCmd,
                         RegistryFQDN,
                         CheckVault,
                         VaultItemName,
                         VaultItemOwner]

    def _parent_resources(self):
        parent_resources = []
        resource_id = self.ctx[ParentLayer.name]
        while resource_id:
            resource = channel.sandbox.get_resource(resource_id)
            parent_resources.append(resource)
            resource_id = resource.attributes.get('parent_layer')
        return parent_resources

    def arcadia_info(self):
        return None, self.ctx[DockerImageName.name], None

    def on_execute(self):
        docker_image_name = self.ctx[DockerImageName.name]
        fqdn = str(self.ctx[RegistryFQDN.name])

        vault_data = ''
        if self.ctx[self.CheckVault.name]:
            logging.info('Acquiring vault key...')
            owner = self.ctx[self.VaultItemOwner.name] or self.author
            vault_data = self.get_vault_data(owner, self.ctx[self.VaultItemName.name])
            if len(vault_data) == 0:
                raise errors.SandboxTaskUnknownError('Vault key acquiring failed')
            logging.info('Vault data: %s...<suppressed>', str(vault_data)[0:10])

        parent_resources = self._parent_resources()
        build_log = open(get_logs_folder() + '/build.log', 'a')

        build_log.write('\n\n=== In the name of f****g docker! ===\n')
        build_log.write(
            '>> During build process portod calls tar with "--sparse" arg and this immutable. '
            'Because it compiled into portod. Docker 1.7 do not support sparse tar-archives. '
            'We _have_to_ extract whole content from this tar archive to FS.\n'
        )
        build_log.write('>> We need to be root to keep permissions. Therefore sudo is here.\n\n')
        build_log.flush()

        if self.ctx[self.CheckVault.name]:
            build_log.write('>> docker login using username "'+self.ctx[self.VaultItemName.name]+'".\n')
            build_log.flush()
            run_process(['docker', 'login', '-u', self.ctx[self.VaultItemName.name], '-p', vault_data, '-e', 'cocaine-admin@yandex-team.ru', fqdn], stdout=build_log, stderr=build_log)
            build_log.flush()

        for resource in reversed(parent_resources):
            sandbox_res_id = fqdn+'/sandbox-resource-id'+':'+str(resource.id)

            build_log.write('>> Pulling layer "'+sandbox_res_id+'" from registry...\n')
            build_log.flush()
            run_process(['docker', 'pull', sandbox_res_id], check=False, stdout=build_log, stderr=build_log)
            sp = run_process(['docker images|grep -Eq '+fqdn+'/sandbox-resource-id'+'[[:blank:]]+'+str(resource.id)], shell=True, check=False, stdout=build_log, stderr=build_log)
            sp.communicate()[0]
            if sp.returncode != 0:
                build_log.write('>> Required layer "'+sandbox_res_id+'" not found, therefore it will be created.\n')
                build_log.flush()
                layer_tarball = self.sync_resource(resource.id)
                repack_dir = self.abs_path('docker-tar-repack')
                build_dir = self.abs_path('build_dir')
                build_log.write('>> Creating dirs: "'+repack_dir+'", "'+build_dir+'"\n')
                build_log.flush()
                run_process(['mkdir', '-p', repack_dir, build_dir], stdout=build_log, stderr=build_log)
                build_log.flush()
                build_log.write('>> Extracting files from archive\n')
                build_log.flush()
                run_process(['sudo', 'tar', '-C', repack_dir, '-xf', layer_tarball], stdout=build_log, stderr=build_log)
                build_log.flush()
                build_log.write('>> Removing files of character devices. Reason: docker with overlay storage driver breaks while image build\n')
                build_log.flush()
                run_process(['sudo', 'find', repack_dir, '-xdev', '-type', 'c', '-exec', 'rm', '-f', '{}', '+'], stdout=build_log, stderr=build_log)
                build_log.flush()
                if resource.attributes.get('parent_layer') is not None:
                    tar_name = build_dir+'/'+str(resource.id)+'.tar.gz'
                    build_log.write('>> Creating archive '+tar_name+'\n')
                    build_log.flush()
                    run_process(['sudo', 'tar', '--numeric-owner', '-zpc', '-f', tar_name, '-C', repack_dir, './'], stdout=build_log, stderr=build_log)
                    build_log.flush()
                    build_log.write('>> Creating dockerfile\n')
                    build_log.flush()
                    dockerfile_path = build_dir+'/Dockerfile'
                    dockerfile_fd = open(dockerfile_path, 'w')
                    dockerfile_fd.write('FROM '+fqdn+'/sandbox-resource-id:'+str(resource.attributes.get('parent_layer'))+'\n')
                    dockerfile_fd.write('MAINTAINER cocaine-admin@yandex-team.ru\n')
                    dockerfile_fd.write('ADD '+str(resource.id)+'.tar.gz /\n')
                    if resource.attributes.get('parent_layer') == self.ctx[ParentLayer.name]:
                        dockerfile_fd.write('CMD '+self.ctx[DockerfileCmd.name]+'\n')
                    dockerfile_fd.close()
                    build_log.write('>> Creating intermediate layer '+sandbox_res_id+' in docker.\n')
                    build_log.flush()
                    build_log.write('>> Invoking docker build\n')
                    build_log.flush()
                    run_process(['docker', 'build', '-t', sandbox_res_id, build_dir], stdout=build_log, stderr=build_log)
                    build_log.flush()
                    build_log.write('>> Removing archive '+tar_name+'\n')
                    build_log.flush()
                    run_process(['sudo', 'rm', '-f', tar_name], stdout=build_log, stderr=build_log)
                    build_log.flush()
                else:
                    build_log.write('>> Creating base layer "'+sandbox_res_id+'" in docker.\n')
                    build_log.flush()
                    run_process(['bash', '-c', 'sudo tar --numeric-owner -pcf - -C '+repack_dir+' .|docker import - '+sandbox_res_id], stdout=build_log, stderr=build_log)
                    build_log.flush()
                build_log.write('>> Pushing layer "'+sandbox_res_id+'" to registry\n')
                build_log.flush()
                run_process(['docker', 'push', sandbox_res_id], stdout=build_log, stderr=build_log)
                build_log.flush()
                build_log.write('>> Removing dirs: "'+repack_dir+'", "'+build_dir+'" \n')
                build_log.flush()
                run_process(['sudo', 'rm', '-rf', repack_dir, build_dir], stdout=build_log, stderr=build_log)
                build_log.flush()

        build_log.write('>> Creating final layer "'+docker_image_name+'" in docker.\n')
        build_log.flush()
        build_log.write('>> Creating dockerfile\n')
        build_log.flush()
        build_dir = self.abs_path('build_dir')
        build_log.write('>> Creating dir: "'+build_dir+'"\n')
        build_log.flush()
        run_process(['mkdir', '-p', build_dir], stdout=build_log, stderr=build_log)
        build_log.flush()
        dockerfile_path = build_dir+'/Dockerfile'
        dockerfile_fd = open(dockerfile_path, 'w')
        dockerfile_fd.write('FROM '+fqdn+'/sandbox-resource-id:'+str(self.ctx[ParentLayer.name])+'\n')
        dockerfile_fd.write('MAINTAINER cocaine-admin@yandex-team.ru\n')
        dockerfile_fd.write('CMD '+self.ctx[DockerfileCmd.name]+'\n')
        dockerfile_fd.close()
        build_log.write('>> Invoking docker build\n')
        build_log.flush()
        run_process(['docker', 'build', '-t', docker_image_name, build_dir], stdout=build_log, stderr=build_log)
        build_log.flush()
        build_log.write('>> Removing dir: "'+build_dir+'" \n')
        build_log.flush()
        run_process(['sudo', 'rm', '-rf', build_dir], stdout=build_log, stderr=build_log)
        build_log.flush()
        # tag = given parameter docker image name + year|month|day|seconds count from 0:00
        tag = fqdn+'/'+docker_image_name+':'+str(time.strftime('%Y%m%d-%H%M%S', time.gmtime(time.time())))
        build_log.write('>> Setting tag "'+tag+'"\n')
        build_log.flush()
        run_process(['docker', 'tag', '-f', docker_image_name, tag], stdout=build_log, stderr=build_log)
        build_log.flush()
        build_log.write('>> Pushing to registry\n')
        build_log.flush()
        run_process(['docker', 'push', tag], stdout=build_log, stderr=build_log)
        build_log.flush()
        if self.ctx[self.CheckVault.name]:
            build_log.write('>> docker logout from "'+fqdn+'".\n')
            build_log.flush()
            run_process(['docker', 'logout', fqdn], stdout=build_log, stderr=build_log)
            build_log.flush()
        build_log.write('>> Removing all local images from docker pool\n')
        build_log.flush()
        run_process(['docker rmi -f $(docker images -q)'], shell=True, check=False, stdout=build_log, stderr=build_log)
        build_log.flush()

        build_log.close()

        self.ctx['registry_image'] = tag

    def on_release(self, additional_parameters):

        target_status = additional_parameters['release_status']
        parent_resources = self._parent_resources()
        for resource in parent_resources:
            if 'released' not in resource.attributes:
                raise Exception('Parent layer {} ({}) is not released yet'.format(resource.description, resource.id))
            resource_status = resource.attributes['released']
            logging.info('Parent layer {} ({}) released to {}'.format(resource.description, resource.id, resource_status))
            if RELEASE_STATUSES.index(resource_status) > RELEASE_STATUSES.index(target_status):
                raise Exception('Parent layer {} ({}) has incompatible release {}'.format(resource.description, resource.id, resource_status))

        SandboxTask.on_release(self, additional_parameters)


__Task__ = CocaineLayersExportToRegistry
