#!/usr/bin/env python2
#  -*- coding: utf-8 -*-

from __future__ import division
from __future__ import unicode_literals
from sandbox.sandboxsdk import environments
from sandbox.sdk2 import yav
import logging
from datetime import datetime, timedelta
import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2


logger = logging.getLogger(__name__)


class SPICollector(sdk2.Task):

    name_for_humans = "Spi stats collector"

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1000  # 1G
        environments = [
            # List of required packages
            environments.PipEnvironment("yandex-yt", version="0.9.29"),
            environments.PipEnvironment("startrek-client", version="1.9.7"),
        ]
        client_tags = task_env.TaskTags.startrek_client

    class Context(sdk2.Task.Context):
        loaded_tickets = []

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Auth') as dash_bundle:
            st_token_owner = sdk2.parameters.String("Startrek token owner")
            st_token_name = sdk2.parameters.String("Startrek token name")
            yt_token_owner = sdk2.parameters.String("YT token owner")
            yt_token_name = sdk2.parameters.String("YT token name")

    def on_execute(self):
        st_secret = yav.Secret(self.Parameters.st_token_secret)
        st_token = st_secret.data()[self.Parameters.st_token_key]
        yt_secret = yav.Secret(self.Parameters.yt_token_secret)
        yt_token = yt_secret.data()[self.Parameters.yt_token_key]
        tickets = self.get_tickets(st_token, yt_token)
        tickets.write_yt()


class TicketListCollector(SPICollector):

    class Parameters(SPICollector.Parameters):
        with sdk2.parameters.Group('SPI Collector parameters') as dash_bundle:
            tickets = sdk2.parameters.List('Tickets')

    def get_tickets(self, st_token, yt_token):
        from sandbox.projects.CollectSpiStat.main.startrek import StartrekTickets
        tickets = StartrekTickets.from_tickets_lst(st_token=st_token, issues_list=self.Parameters.tickets)
        self.Context.loaded_tickets.append(self.Parameters.tickets)
        return tickets


class TicketPeriodCollector(SPICollector):
    class Parameters(SPICollector.Parameters):
        with sdk2.parameters.Group('SPI Collector parameters') as dash_bundle:
            last_week = sdk2.parameters.Bool("Last week")
            with last_week.value[False]:
                start = sdk2.parameters.String('Start at')
                end = sdk2.parameters.String('End at')

    def get_tickets(self, st_token, yt_token):
        from sandbox.projects.CollectSpiStat.main.startrek import StartrekTickets
        # from search.mon.stater.contrib.datemath.datemath import parse
        # start = parse(self.Parameters.start)
        # end = parse(self.Parameters.end)
        if self.Parameters.last_week:
            start = (datetime.now() - timedelta(days=7)).date()
        else:
            start = datetime.strprime(self.Parameters.start, '%Y-%m-%d')
            end = None
            if self.Parameters.end:
                try:
                    end = datetime.strprime(self.Parameters.end, '%Y-%m-%d')
                except:
                    end = None
        tickets = StartrekTickets.from_period(st_token=st_token, start_date=start, end_date=end)
        return tickets
