#!/usr/bin/env python
# -*- coding: utf-8 -*-
from startrek_client import Startrek
from datetime import datetime, timedelta
from pytz import timezone
from startrek_client.objects import Resource
from startrek_client.exceptions import Forbidden
import os
import re
import logging
import yt.wrapper as yt


__logger__ = logging.getLogger(__name__)


class StartrekTicket(object):
    DEFAULT_REACTION_TIME = timedelta(minutes=5)

    def __init__(self, issue):
        self._issue = issue
        self.duty_timeline_pattern = re.compile(r'\s*[0-9]?[0-9]:[0-9]{2}')
        self._has_timeline = None
        self._has_diagnostics = None
        self._has_iconostasis_screenshots = None
        self._has_auto_begin_time = None
        self._duty_actions_charts_images = None
        self._duty_actions_charts_links = None
        # Wait for https://st.yandex-team.ru/YAINCBOT-345
        self._start_timeline_accounting_at = datetime(2020, 12, 1, tzinfo=timezone('Europe/Moscow'))
        self._has_auto_notification_time = None
        self._qualitative = None
        self._troubles = None
        self._caller = None
        self._action_items = None

    @property
    def issue(self):
        return self._issue

    @property
    def author(self):
        return self._issue.createdBy.id

    @property
    def created_at(self):
        return datetime.strptime(self._issue.createdAt, '%Y-%m-%dT%H:%M:%S.%f%z')

    @property
    def incident_start(self):
        start_time = self._issue.sroBeginTime or self._issue.sreBeginTime or self._issue.createdAt
        if start_time is not None:
            return datetime.strptime(start_time, '%Y-%m-%dT%H:%M:%S.%f%z')

    @property
    def notified_at(self):
        notify_time = self._issue.sroNotificationTime or self._issue.createdAt
        if notify_time is not None:
            return datetime.strptime(notify_time, '%Y-%m-%dT%H:%M:%S.%f%z')

    @property
    def key(self):
        return self._issue.key

    @property
    def tags(self):
        return self._issue.tags

    @property
    def action_items(self):
        if self._action_items is not None:
            return self._action_items
        result = []
        for link in self._issue.links:
            try:
                if 'spi:actionitem' in link.object.tags:
                    result.append(link.object)
            except Forbidden:
                __logger__.error('Access forbidden for potential AI %s', link.object.key)
        self._action_items = result.copy()
        return self._action_items

    @property
    def action_items_keys(self):
        return [t.key for t in self.action_items]

    @property
    def unmonitored(self):
        return 'unmonitored' in self.tags or 'monitoring:none' in self.tags

    @property
    def duty_actions_charts_images(self):
        if self._duty_actions_charts_images is not None:
            return self._duty_actions_charts_images

        return self._duty_actions_charts_images or []

    @property
    def duty_actions_charts_links(self):
        if self._duty_actions_charts_links is not None:
            return self._duty_actions_charts_links

        return self._duty_actions_charts_links or []

    @property
    def duty_actions(self):
        description = self._issue.description
        if description:
            return description.split(u'== Действия дежурного ==')[-1].split(u'==Анализ==')[0].strip()
        else:
            return ''

    @property
    def duty_timeline(self):
        return (len(self.duty_timeline_pattern.findall(self.duty_actions)) > 0) or self.has_timeline

    @property
    def false_positive(self):
        return 'false:positive' in self.tags or 'monitoring:falsepositive' in self.tags

    @property
    def match_sla(self):
        return self.reaction_time <= self.DEFAULT_REACTION_TIME

    @property
    def environment(self):
        for tag in self.tags:
            if tag.startswith('env:'):
                return ':'.join(tag.split(':')[1:])

    @property
    def components(self):
        return [c.name for c in self._issue.components]

    @property
    def vertical(self):
        components = self._issue.components
        if components:
            return self._issue.components[0].name
        else:
            __logger__.critical('Ticket %s has no components set', self.key)
            vertical = '-'.join(self.product.split('-')[1:]).lower()
            return vertical

    @property
    def service(self):
        for tag in self.tags:
            if tag.startswith('service:'):
                return (':'.join(tag.split(':')[1:])).lower()
        return ''

    @property
    def services(self):
        result = []
        for tag in self.tags:
            if tag.startswith('service:'):
                result.append((':'.join(tag.split(':')[1:])).lower())
        return result

    @property
    def product(self):
        for tag in self.tags:
            if tag.startswith('product:'):
                return ':'.join(tag.split(':')[1:])
        return ''

    @property
    def has_begin_time(self):
        return self._issue.sroBeginTime is not None

    @property
    def has_notification_time(self):
        return self._issue.sroNotificationTime is not None

    @property
    def incorrect_resolution(self):
        return self._issue.resolution.name == 'Дубликат' or self._issue.resolution.name == 'Некорректный'

    @property
    def reaction_time(self):
        result = timedelta(0)
        if not self.negative_reaction_time:
            result = self.notified_at - self.incident_start
        return result

    @property
    def negative_reaction_time(self):
        return self.notified_at < self.incident_start

    @property
    def fielddate(self):
        return self.created_at.date()

    @property
    def st_link(self):
        return 'https://st.yandex-team.ru/{issue}'.format(issue=self.key)

    @property
    def caller(self):
        return self._caller

    @property
    def has_timeline(self):
        if self._has_timeline is not None:
            return self._has_timeline
        assign_comments = [
            c for c in self._issue.comments if
            c.createdBy.id == 'robot-searchmon' and (
                c.text.startswith('====Принято в работу') or
                c.text.startswith('<{Таймлайн действий') or
                ' установил SRO Notification Time из сообщения' in c.text
            )
        ]
        __logger__.info('Found %s assignment comments', len(assign_comments))
        if assign_comments:
            self._has_timeline = True
            matched = re.findall(
                '^staff:([^ ]+) (?:призвал дежурных.|установил SRO Notification Time из сообщения:)$',
                assign_comments[0].text,
                flags=re.MULTILINE
            )
            if matched:
                self._caller = matched[0]
            else:
                splitted = re.split(
                    '\\[[0-9.]{8} [0-9:]{8}\\]',
                    assign_comments[0].text
                )
                if len(splitted) > 1:
                    splitted = splitted[1].split('\n')
                    if splitted:
                        splitted = splitted[1]
                        matched = re.findall(
                            '(?<=^\\* Сделал staff:)[^.]+',
                            splitted
                        )
                        if matched:
                            self._caller = matched[0]

            return True
        self._has_timeline = False
        return False

    @property
    def has_auto_notification_time(self):
        if self._has_auto_notification_time is not None:
            return self._has_auto_notification_time
        # TODO: add timeline control to ensure that SRO Notification Time exactly matches value that YaIncBot set
        history = self._issue.changelog
        tickenator_actions = [x for x in history if x.updatedBy.id == 'robot-searchmon']
        for action in tickenator_actions:
            for changed_field in action.fields:
                if 'field' not in changed_field:
                    continue
                if changed_field['field'].id == 'sroNotificationTime':
                    self._has_auto_notification_time = True
                    return True
        self._has_auto_notification_time = False
        return False

    @property
    def has_diagnostics(self):
        if self._has_diagnostics is not None:
            return self._has_diagnostics

        diagnostics_comments = [
            c for c in self._issue.comments if
            c.createdBy.id == 'robot-tickenator' and c.text.startswith('<{Графики доли')
        ]
        __logger__.info('Found %s assignment comments', len(diagnostics_comments))
        if diagnostics_comments:
            self._has_diagnostics = True
            return True
        self._has_diagnostics = False
        return False

    @property
    def has_iconostasis_screenshots(self):
        if self._has_iconostasis_screenshots is not None:
            return self._has_iconostasis_screenshots

        diagnostics_comments = [
            c for c in self._issue.comments if
            c.createdBy.id == 'robot-tickenator' and c.text.startswith('<{Скриншоты иконостаса')
        ]
        __logger__.info('Found %s assignment comments', len(diagnostics_comments))
        if diagnostics_comments:
            self._has_iconostasis_screenshots = True
            return True
        self._has_iconostasis_screenshots = False
        return False

    @property
    def has_auto_begin_time(self):
        if self._has_auto_begin_time is not None:
            return self._has_auto_begin_time
        # TODO: add timeline control to ensure that SRO Begin Time exactly matches value that tickenator set
        history = self._issue.changelog
        tickenator_actions = [x for x in history if x.updatedBy.id == 'robot-tickenator']
        for action in tickenator_actions:
            for changed_field in action.fields:
                if 'field' not in changed_field:
                    continue
                if changed_field['field'].id == 'sroBeginTime':
                    self._has_auto_begin_time = True
                    return True
        self._has_auto_begin_time = False
        return False

    @property
    def qualitative(self):
        if self._qualitative is not None:
            return self._qualitative
        self._qualitative = False
        if (
                self.has_begin_time and
                self.has_notification_time and
                self.duty_actions and
                self.duty_timeline and
                (
                    (
                        self.has_auto_notification_time and
                        self.created_at >= self._start_timeline_accounting_at
                    ) or
                    self.created_at < self._start_timeline_accounting_at
                )
        ):
            self._qualitative = True
        return self._qualitative

    @property
    def troubles(self):
        if self._troubles is not None:
            return self._troubles
        result = []
        if not self.has_begin_time:
            result.append('Не задан sroBeginTime')
        if not self.has_notification_time:
            result.append('Не задан sroNotificationTime')
        if not self.duty_timeline:
            result.append('Нет действий деж.смены')
        if self.created_at >= self._start_timeline_accounting_at and not self.has_auto_notification_time:
            result.append('Нет автоматически проставленного sroNotificationTime')
        self._troubles = result.copy()

        return ', '.join(self._troubles)

    @property
    def good(self):
        return self.qualitative and self.match_sla

    @property
    def resolution(self):
        if self._issue.resolution is not None:
            return self._issue.resolution.name
        else:
            return None

    @property
    def devops_acquire_time(self):
        if self._issue.sroNotificationTime is None or self._issue.notificationTime is None:
            return None
        return max([
            0,
            datetime.strptime(
                self._issue.notificationTime, '%Y-%m-%dT%H:%M:%S.%f%z'
            ).timestamp() - datetime.strptime(
                self._issue.sroNotificationTime, '%Y-%m-%dT%H:%M:%S.%f%z'
            ).timestamp()
        ])

    @property
    def ydt(self):
        return self._issue.ydt

    @property
    def need_disaster(self):
        result = None
        nd_tag = [tag for tag in self.tags if tag.startswith('need_disaster:')]
        if nd_tag:
            result = False if nd_tag[0] == 'need_disaster:false' else True
        return result

    @property
    def disaster_created(self):
        return 'disaster_created' in self.tags

    @property
    def juggler_host(self):
        result = None
        jh_tag = [tag for tag in self.tags if tag.startswith('juggler_host:')]
        if jh_tag:
            jh_tag_kv = jh_tag[0].split(':')
            if len(jh_tag_kv) > 1:
                result = jh_tag_kv[1]
        return result

    @property
    def juggler_service(self):
        result = None
        js_tag = [tag for tag in self.tags if tag.startswith('juggler_service:')]
        if js_tag:
            js_tag_kv = js_tag[0].split(':')
            if len(js_tag_kv) > 1:
                result = js_tag_kv[1]
        return result

    @property
    def disaster_ai(self):
        result = [t.key for t in self.action_items if 'disaster_alert' in t.tags]
        if result:
            return result[0]
        else:
            return None

    @property
    def chyt_json(self):
        sro_begin_time = int(datetime.strptime(
            self._issue.sroBeginTime, '%Y-%m-%dT%H:%M:%S.%f%z'
        ).timestamp()) if self._issue.sroBeginTime else None
        sro_notification_time = int(datetime.strptime(
            self._issue.sroNotificationTime, '%Y-%m-%dT%H:%M:%S.%f%z'
        ).timestamp()) if self._issue.sroNotificationTime else None
        sro_end_time = int(datetime.strptime(
            self._issue.sroEndTime, '%Y-%m-%dT%H:%M:%S.%f%z'
        ).timestamp()) if self._issue.sroEndTime else None
        sre_begin_time = int(datetime.strptime(
            self._issue.sreBeginTime, '%Y-%m-%dT%H:%M:%S.%f%z'
        ).timestamp()) if self._issue.sreBeginTime else None
        sre_notification_time = int(datetime.strptime(
            self._issue.notificationTime, '%Y-%m-%dT%H:%M:%S.%f%z'
        ).timestamp()) if self._issue.notificationTime else None
        sre_end_time = int(datetime.strptime(
            self._issue.sreEndTime, '%Y-%m-%dT%H:%M:%S.%f%z'
        ).timestamp()) if self._issue.sreEndTime else None

        return {
            'key': self.key,
            'author': self.author,
            'caller': self.caller,
            'unmonitored': self.unmonitored,
            'action_items': self.action_items_keys,
            'need_disaster': self.need_disaster,
            'disaster_ai': self.disaster_ai,
            'disaster_created': self.disaster_created,
            'juggler_host': self.juggler_host,
            'juggler_service': self.juggler_service,
            'has_duty_actions': self.duty_timeline,
            'false_positive': self.false_positive,
            'priority': self._issue.priority.id,
            'match_sla': self.match_sla,
            'environment': self.environment,
            'vertical': self.vertical,
            'service': self.service,
            'product': self.product,
            'components': [c.name for c in self._issue.components],
            'services': self.services,
            'resolution': self.resolution or '',
            'has_timeline': self.has_timeline,
            'has_diagnostics': self.has_diagnostics,
            'has_charts_images_in_duty_actions': len(self.duty_actions_charts_images) > 0,
            'has_charts_links_in_duty_actions': len(self.duty_actions_charts_links) > 0,
            'created': int(self.created_at.timestamp()),
            'sro_begin_time': sro_begin_time,
            'sro_notification_time': sro_notification_time,
            'sro_end_time': sro_end_time,
            'sre_begin_time': sre_begin_time,
            'sre_notification_time': sre_notification_time,
            'sre_end_time': sre_end_time,
            'start': int(self.incident_start.timestamp()),
            'notified': int(self.notified_at.timestamp()),
            'marty_reaction_time': float(self.reaction_time.total_seconds()),
            'good': self.good,
            'qualitative': self.qualitative,
            'devops_acquire_time': float(
                self.devops_acquire_time) if self.devops_acquire_time is not None else None,
            'ydt': self.ydt,
            'auto': 'beholder:created' in self._issue.tags,
            'impact': self._issue.impact,
            'tags': self._issue.tags
        }


class DatePeriod(object):
    def __init__(self, start_date, end_date):
        if isinstance(start_date, datetime):
            start_date = start_date.date()
        if isinstance(end_date, datetime):
            end_date = end_date.date()
        self._start_date = start_date
        self._end_date = end_date

    @property
    def start(self):
        return self._start_date

    @property
    def end(self):
        return self._end_date

    def __contains__(self, some_date):
        if isinstance(some_date, datetime):
            some_date = some_date.date()
        if self._end_date is not None:
            return self._start_date <= some_date <= self._end_date
        else:
            return self._start_date <= some_date

    @property
    def startrek(self):
        result = {'from': self._start_date.strftime('%Y-%m-%dT%H:%M:%S.%f%z')}
        if self._end_date is not None:
            result['to'] = self._end_date.strftime('%Y-%m-%dT%H:%M:%S.%f%z')
        return result


class StartrekTickets(object):
    def __init__(
            self, st_token,
            yt_token=None,
            queue='SPI'
    ):
        self._client = Startrek(useragent='stater', token=st_token)
        self._queue = queue
        self.set_yt_client(token=yt_token)
        self._yt_table_path = "//home/searchmon/stats/spi"
        self._yt_client = None
        self._issues = {}

    def append(self, issue):
        if not isinstance(issue, StartrekTicket):
            if not isinstance(issue, Resource):
                issue = self._client.issues[issue]
            issue = StartrekTicket(issue=issue)
        self._issues[issue.key] = issue

    @classmethod
    def from_tickets_lst(cls, st_token, yt_token, issues_list):
        result = cls(st_token=st_token, yt_token=yt_token)
        for issue in issues_list:
            result.append(issue)
        return result

    @classmethod
    def from_period(cls, st_token, yt_token, start_date, end_date=None):
        result = cls(st_token=st_token, yt_token=yt_token)
        result._period = DatePeriod(start_date, end_date)
        result._load_tickets()
        return result

    def _load_tickets(self):
        issues_list = self._client.issues.find(
            filter={'queue': self._queue, 'created': self._period.startrek},
            per_page=10
        )
        for issue in issues_list:
            self._issues[issue.key] = StartrekTicket(issue=issue)

    def get(self, ticket_id):
        return self._issues.get(ticket_id)

    def set_yt_client(self, token=None):
        if token is None:
            token=os.environ['YT_TOKEN']
        self._yt_client = yt.YtClient(proxy="hahn", token=token)

    @property
    def yt_client(self):
        if self._yt_client is None:
            self.set_yt_client()
        return self._yt_client

    @property
    def yt_table(self):
        result = []
        for ticket in self._issues.values():
            result.append(ticket.chyt_json)
        return sorted(result, key=lambda x: x['key'])

    def write_yt(self):
        self.yt_client.write_table(
            self._yt_table_path,
            self.merge_tables(self.current_table, self.yt_table),
            format=yt.JsonFormat(attributes={"encode_utf8": False})
        )

    @property
    def current_table(self):
        return list(self.yt_client.read_table(
            self._yt_table_path,
            format=yt.JsonFormat(attributes={"encode_utf8": False}))
        )

    @staticmethod
    def merge_tables(current_table, new_table):
        new_dict = {x['key']: x for x in new_table}
        cur_dict = {x['key']: x for x in current_table}
        cur_dict.update(new_dict)
        return sorted(list(cur_dict.values()), key=lambda x: x['key'])
