import os
import shutil
import logging

from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import (
    DictRepeater,
    ListRepeater,
    SandboxBoolParameter,
    SandboxStringParameter,
    LastReleasedResource,
)

from sandbox.sandboxsdk.paths import get_unique_file_name, make_folder
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common.userdata import mr_base_task


class StatesResources(LastReleasedResource):
    """
        Join these states:
    """
    name = 'state_resource_ids'
    description = "Resources with stored tables:"
    resource_type = [resource_types.USERDATA_TABLES_ARCHIVE]
    multiple = True
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


class ResultDescr(SandboxStringParameter):
    name = 'result_name'
    description = 'Name of resulting state:'
    required = False
    default_value = None
    group = mr_base_task.OUTPUT_PARAMS_GROUP_NAME


class MergeResourceAttrs(SandboxBoolParameter):
    name = 'merge_resource_attrs'
    description = 'Merge resources\' attributes:'
    required = False
    default_value = True
    group = mr_base_task.MISC_PARAMS_GROUP_NAME


class ExtraAttrs(DictRepeater, SandboxStringParameter):
    name = 'extra_attrs'
    description = "Add attributes to resulting resource (will be formatted via task context):"
    required = False
    group = mr_base_task.OUTPUT_PARAMS_GROUP_NAME


class DropAttrs(ListRepeater, SandboxStringParameter):
    name = 'drop_attrs'
    description = "Drop these attributes from merged:"
    required = False
    group = mr_base_task.OUTPUT_PARAMS_GROUP_NAME


class CombineMRTables(SandboxTask):
    """
        Take several USERDATA_TABLES_ARCHIVE resources and create one with all the tables
    """
    type = "COMBINE_MR_TABLES"

    input_parameters = [
        StatesResources,
        MergeResourceAttrs,
        ResultDescr,
        ExtraAttrs,
        DropAttrs,
    ]

    client_tags = Tag.LINUX_PRECISE
    required_ram = 23 << 10
    execution_space = 40000

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self
        tables_resource = self.create_resource(
            description=self.ctx.get(ResultDescr.name, "<no descr>"),
            resource_path='',
            resource_type=resource_types.USERDATA_TABLES_ARCHIVE,
            arch='any',
        )

        self.ctx['tables_resource_id'] = tables_resource.id

    def get_tables_prefix(self):
        return "sandbox/"

    def on_execute(self):
        def reprefix(n, op, np):
            if np is None:
                return n
            np = np.replace('/', ':')
            return np + n[len(op):]

        respath = get_unique_file_name('', 'result')
        make_folder(respath)
        attrs = {}
        inres = self.ctx[StatesResources.name]
        if hasattr(inres, '__iter__'):
            pass
        elif ',' in inres:
            inres = inres.split(',')
        else:
            inres = [inres]

        for rid in inres:
            path = self.sync_resource(rid)
            res = channel.sandbox.get_resource(rid)
            if self.ctx.get(MergeResourceAttrs.name):
                attrs.update(res.attributes)
            files = [i for i in os.listdir(path) if i.endswith('.lenval.gz')]
            logging.debug('resource {rid} has files: {files}'.format(rid=rid, files=files))
            for f in files:
                target_prefix_name = reprefix(f, res.attributes['tables_prefix'], self.get_tables_prefix())
                # XXX should I warn/fail on duplicates?
                shutil.copyfile(os.path.join(path, f), os.path.join(respath, target_prefix_name))

        extra = self.ctx.get(ExtraAttrs.name)
        if extra and isinstance(extra, dict):
            for k, v in extra.items():
                if isinstance(v, str):
                    attrs[k.format(**self.ctx)] = v.format(**self.ctx)
                else:
                    attrs[k.format(**self.ctx)] = v

        drop_attrs = ['ttl', 'do_not_remove', 'backup_task', 'release', 'mds']
        drop_extra = self.ctx.get(DropAttrs.name)
        if drop_extra and isinstance(drop_extra, list):
            drop_attrs += drop_extra

        for k in drop_attrs:
            if k in attrs:
                del attrs[k]

        attrs['tables_prefix'] = self.get_tables_prefix()

        resource = channel.sandbox.get_resource(self.ctx['tables_resource_id'])
        for n, v in attrs.iteritems():
            channel.sandbox.set_resource_attribute(resource.id, n, v)
        self.change_resource_basename(resource.id, respath)
        self.mark_resource_ready(resource.id)


__Task__ = CombineMRTables
