# -*- coding: utf-8 -*-

from sandbox import common
import logging
import os

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import environments

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.noapacheupper import request
from sandbox.projects.common.noapacheupper.diff import app_host as ahd
from sandbox.projects.common.noapacheupper import response_patcher as nrp
from sandbox.projects.websearch.upper import resources as upper_resources
from sandbox.projects.websearch import release_setup

INPUT_RESOURCES_TYPES = [
    upper_resources.NoapacheAdjusterResponses,
    upper_resources.NoapacheBlenderResponses,
    resource_types.RESPONSES_SEARCH_APP_SERVICE,
]

QUERIES_PARAM_FOR_TASK = {
    'GET_STANDALONE_NOAPACHEUPPER_RESPONSES': 'noapache_requests_resource_id',
}

NOAPACHE = 'noapache'
ADJUSTER = 'adjuster'
BLENDER = 'blender'


class Params:
    class ResponsesType(sp.SandboxSelectParameter):
        name = 'responses_type'
        description = 'Responses type'
        choices = [
            (NOAPACHE, NOAPACHE),
            (ADJUSTER, ADJUSTER),
            (BLENDER, BLENDER),
        ]
        default_value = NOAPACHE
        resource_type = INPUT_RESOURCES_TYPES

    class Responses1(sp.ResourceSelector):
        name = 'responses1_resource_id'
        description = 'Responses #1'
        resource_type = INPUT_RESOURCES_TYPES

    class Responses2(sp.ResourceSelector):
        name = 'responses2_resource_id'
        description = 'Responses #2'
        resource_type = INPUT_RESOURCES_TYPES

    class ToolConverter(sp.ResourceSelector):
        name = 'tool_converter'
        description = 'Tool to convert various apphost protocol messages to/out of json'
        resource_type = resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE

    class ToolProto2Hr(sp.ResourceSelector):
        name = 'tool_proto2hr'
        description = 'Tool for convert raw proto to proto-hr format'
        resource_type = resource_types.PROTO2HR_EXECUTABLE

    class AppHostOps(sp.ResourceSelector):
        name = 'app_host_ops'
        description = 'New tool to convert various apphost protocol messages in/out of json (app_host_ops)'
        resource_type = upper_resources.APP_HOST_OPS_EXECUTABLE

    class DiffsPerFile(sp.SandboxIntegerParameter):
        name = 'diffs_per_file'
        description = 'Records per single diff file'
        default_value = 10

    class ResponsesLimit(sp.SandboxIntegerParameter):
        name = 'responses_limit'
        description = 'Limit number of used responses (0 = all)'
        default_value = 0

    class UseIndexForComparison(sp.SandboxBoolParameter):
        """Allows to save and compare async binary responses with index"""
        name = 'use_index_for_comparison'
        description = 'Use index for comparison'
        default_value = True

    class MaxFileSizeDifference(sp.SandboxFloatParameter):
        name = 'max_file_size_difference'
        description = 'Threshold for data file size difference in percentages (fill 0.0 if you don\'t want to check size)'
        default_value = 0.0

    lst = (
        ResponsesType,
        Responses1,
        Responses2,
        ToolConverter,
        ToolProto2Hr,
        AppHostOps,
        DiffsPerFile,
        ResponsesLimit,
        UseIndexForComparison,
        MaxFileSizeDifference,
    )


class CompareApphostServiceResponses(bugbanner.BugBannerTask):
    """
        Сравнивает два выхлопа сервиса, поддерживающего протокол apphost.
        Строит diff по запросам и по изменившимся свойствам.
    """

    type = 'COMPARE_APPHOST_SERVICE_RESPONSES'
    input_parameters = Params.lst

    environments = [
        environments.PipEnvironment("yandex-blender-factor-storage"),
    ]

    client_tags = release_setup.WEBSEARCH_TAGS_P1

    compare_result_id = 'out_resource_id'

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self.create_resource(
            self.descr, 'compare_result',
            resource_types.BASESEARCH_RESPONSES_COMPARE_RESULT,
            attributes={
                'backup_task': True,
                'ttl': 14,
            },
        )
        self.ctx[self.compare_result_id] = resource.id
        self.ctx['kill_timeout'] = 8 * 60 * 60

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.NoapacheUpper)

        responses1_resource_id = self.ctx[Params.Responses1.name]
        responses2_resource_id = self.ctx[Params.Responses2.name]
        queries = self._get_queries(responses1_resource_id, responses2_resource_id)

        data_dir1 = channel.task.sync_resource(responses1_resource_id)
        data_dir2 = channel.task.sync_resource(responses2_resource_id)
        data_filename1 = os.path.join(data_dir1, 'data')
        data_filename2 = os.path.join(data_dir2, 'data')
        index_filename1 = os.path.join(data_dir1, 'index')
        index_filename2 = os.path.join(data_dir2, 'index')

        # SEARCH - 5388
        threshold = utils.get_or_default(self.ctx, Params.MaxFileSizeDifference)
        if threshold:
            data_size1 = os.path.getsize(data_filename1)
            data_size2 = os.path.getsize(data_filename2)
            if not data_size1 or not data_size2:
                self.set_info("Responses data file is empty", do_escape=False)
                size_diff = True
            else:
                size_increasing = abs(data_size1 - data_size2) * 100.0 / min(data_size1, data_size2)
                if size_increasing > threshold:
                    self.set_info(
                        "There is data size difference equal to {:6f}%".format(size_increasing),
                        do_escape=False
                    )
                    size_diff = True
                else:
                    size_diff = False
                logging.info("Size of data1 is %s", data_size1)
                logging.info("Size of data2 is %s", data_size2)
                logging.info("Difference percentage is %.6f%%", size_increasing)
        else:
            size_diff = False

        converter = channel.task.sync_resource(self.ctx[Params.ToolConverter.name])
        proto2hr = channel.task.sync_resource(self.ctx[Params.ToolProto2Hr.name])
        app_host_ops = channel.task.sync_resource(self.ctx[Params.AppHostOps.name])
        response_patchers = []
        func_can_compare = None
        response_type = utils.get_or_default(self.ctx, Params.ResponsesType)
        response_patchers = [nrp.remove_noapache_specific_noise]
        func_can_compare = nrp.can_compare_noapache_apphost

        if response_type == BLENDER:
            response_patchers.insert(0, nrp.remove_search_errors_debug)  # TODO(avitella@) SEARCH-5251
        response_patchers.append(nrp.patch_dynamic_factors)  # SEARCH-4846
        response_patchers.append(nrp.patch_factors)  # SEARCH-4846
        response_patchers.append(nrp.patch_compressed_factors)  # NOAPACHE-256
        diff_maker = ahd.DiffApphostResponses(
            converter,
            proto2hr,
            response_patchers,
            func_can_compare,
            app_host_ops,
        )
        compare_result_resource = channel.sandbox.get_resource(self.ctx[self.compare_result_id])
        compare_result = compare_result_resource.path
        paths.make_folder(compare_result)
        has_diff = diff_maker.process(
            data_filename1,
            index_filename1,
            data_filename2,
            index_filename2,
            compare_result,
            descr=queries,
            recs_per_file=utils.get_or_default(self.ctx, Params.DiffsPerFile),
            recs_limit=utils.get_or_default(self.ctx, Params.ResponsesLimit),
            use_index=utils.get_or_default(self.ctx, Params.UseIndexForComparison),
        )
        # True == no diff
        self.ctx['compare_result'] = not (has_diff or size_diff)
        if has_diff:
            self.set_info("There are <a href={}>diff</a>".format(compare_result_resource.proxy_url), do_escape=False)
        else:
            fu.write_file(os.path.join(compare_result, 'no_diff'), "no diff")

    def _get_queries(self, responses1_resource_id, responses2_resource_id):
        logging.info("_get_queries via responses resources")
        queries1_resource_id = self._find_queries(responses1_resource_id)
        queries2_resource_id = self._find_queries(responses2_resource_id)
        if queries1_resource_id != queries2_resource_id:
            eh.check_failed("queries for responses1 and responses2 are different")
        if queries1_resource_id:
            return self._load_queries(queries1_resource_id)

    def _find_queries(self, responses_resource_id):
        logging.info("_find_queries queries via responses resource -> get responses task")
        responses_resource = channel.sandbox.get_resource(responses_resource_id)
        task_id = responses_resource.task_id
        logging.info("Retrieving responses getting task, id=%s", task_id)
        sandbox = common.rest.Client()
        task_type = sandbox.task[task_id][:]["type"]
        ctx_pname = QUERIES_PARAM_FOR_TASK.get(task_type)
        if ctx_pname is None:
            return None
        queries_resource_id = sandbox.task[task_id].context[:].get(ctx_pname)
        if not queries_resource_id:
            return None
        if utils.get_or_default(self.ctx, Params.ResponsesType) in [NOAPACHE, BLENDER, ADJUSTER]:
            return request.get_users_queries_for(queries_resource_id)
        return queries_resource_id

    def _load_queries(self, res_id):
        if self.ctx[Params.ResponsesType.name] in [NOAPACHE, BLENDER, ADJUSTER]:
            return self._list_resource_lines(res_id)
        return None

    def _list_resource_lines(self, res_id):
        res_path = self.sync_resource(res_id)
        with open(res_path) as f:
            return [line.strip() for line in f]


__Task__ = CompareApphostServiceResponses
