# -*- coding: utf-8 -*-

import os
import filecmp

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.base_search_quality import dolbilka_response_diff
from sandbox.projects.common.base_search_quality import dolbilka_response_parser as drp

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.channel import channel


_resource_types = [
    'DBG_DOLBILKA_OUTPUT',
    'IDX_OPS_OUTPUT',
]


class DolbilkaOutput1(sp.LastReleasedResource):
    name = 'dolbilka_output1_resource_id'
    description = 'output 1'
    resource_type = _resource_types
    group = 'Task resources'


class DolbilkaOutput2(sp.LastReleasedResource):
    name = 'dolbilka_output2_resource_id'
    description = 'output 2'
    resource_type = _resource_types
    group = 'Task resources'


class FactorNamesFromOutput2(sp.SandboxBoolParameter):
    name = 'factor_names_from_output2'
    description = "Factor names from output 2"
    group = 'Options'
    default_value = True


class DebugMode(sp.SandboxBoolParameter):
    name = 'debug_mode'
    description = "Debug mode"
    group = 'Options'
    default_value = False


class DifferentQueriesMode(sp.SandboxBoolParameter):
    name = 'different_queries_mode'
    description = "Different queries mode"
    group = 'Options'
    default_value = False


class CompareDbgDolbilkaOutputs(bugbanner.BugBannerTask):
    """
        Сравнивает два выхлопа ``dbg_dolbilka`` или ``idx_ops`` в режиме ``serp``.
    """
    type = 'COMPARE_DBG_DOLBILKA_OUTPUTS'

    input_parameters = (
        DolbilkaOutput1,
        DolbilkaOutput2,
        FactorNamesFromOutput2,
        DebugMode,
        DifferentQueriesMode,
    )

    def on_enqueue(self):
        bugbanner.BugBannerTask.on_enqueue(self)
        resource = self._create_resource(self.descr, 'compare_result', resource_types.DBG_DOLBILKA_COMPARE_RESULT)
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.WebBaseSearch)
        output1 = self._read_resource(self.ctx[DolbilkaOutput1.name])
        output2 = self._read_resource(self.ctx[DolbilkaOutput2.name])

        if self.ctx[FactorNamesFromOutput2.name]:
            factor_names_file_name = os.path.join(output2.abs_path(), 'factor_names.txt')
        else:
            factor_names_file_name = os.path.join(output1.abs_path(), 'factor_names.txt')

        dolbilka_output1_file_name = os.path.join(output1.abs_path(), 'output.txt')
        dolbilka_output2_file_name = os.path.join(output2.abs_path(), 'output.txt')

        user_requests2_file_name = None
        if not self.ctx[DebugMode.name]:
            user_requests1_file_name = self._read_resource(
                channel.sandbox.get_task(output1.task_id).ctx['queries_resource_id']).abs_path()
            user_requests2_file_name = self._read_resource(
                channel.sandbox.get_task(output2.task_id).ctx['queries_resource_id']).abs_path()

            if (
                not self.ctx[DifferentQueriesMode.name] and
                not filecmp.cmp(user_requests1_file_name, user_requests2_file_name)
            ):
                self.ctx['not_compatible_data'] = True
                eh.check_failed(
                    "dbg_dolbilka's outputs cannot be compared because they are based on different queries. "
                )

        resource = self._read_resource(self.ctx['out_resource_id'], sync=False)
        paths.make_folder(resource.abs_path())

        eh.verify(
            output1.type.name in ['DBG_DOLBILKA_OUTPUT', 'IDX_OPS_OUTPUT'],
            "Unknown input resource type: " + output1.type.name
        )
        is_dbg_dolbilka = (output1.type.name == 'DBG_DOLBILKA_OUTPUT')

        dolbilka_output1 = drp.parse_and_group_by_query(dolbilka_output1_file_name, is_dbg_dolbilka)
        dolbilka_output2 = drp.parse_and_group_by_query(dolbilka_output2_file_name, is_dbg_dolbilka)

        diffs = dolbilka_response_diff.CompareResults(dolbilka_output1, dolbilka_output2)

        if not diffs.HasDifferences():
            no_differences_file = open(os.path.join(resource.abs_path(), 'noDifferences.txt'), 'w')
            print >>no_differences_file, 'There is no any differences between dbg_dolbilka outputs'
            no_differences_file.close()

            changed_factors = []
        else:
            factor_names = open(factor_names_file_name).read().split('\n')

            if not self.ctx[DebugMode.name]:
                user_requests = open(user_requests2_file_name).read().split('\n')
            else:
                user_requests = ["query" + str(i) for i in range(max(dolbilka_output1.keys()))]

            dolbilka_response_diff.WriteDiffs(diffs, factor_names, user_requests, resource.abs_path())

            different_factors = diffs.GetDifferentFactors()

            changed_factors = []

            for factorNumber in sorted(different_factors.keys()):
                changed_factors.append("FACTOR_{} ({})".format(
                    factorNumber,
                    dolbilka_response_diff.GetFactorName(factor_names, factorNumber)
                ))

            if diffs.HasNewFoundOrLostDocuments():
                changed_factors.append("DOCUMENTS")

        # this is needed for the sandbox test environment
        self.ctx['changed_factors'] = changed_factors

        size1 = os.path.getsize(dolbilka_output1_file_name)
        size2 = os.path.getsize(dolbilka_output2_file_name)

        eh.ensure(size1 != 0 and size2 != 0, "Incorrect file size")

        self.ctx['output_diff'] = 1.0 * size1 / size2

        resource.mark_ready()

    def get_short_task_result(self):
        if self.is_completed():
            if len(self.ctx['changed_factors']) > 0:
                return "diff"
            else:
                return "no diff"
        return None


__Task__ = CompareDbgDolbilkaOutputs
