# -*- coding: utf-8 -*-

import os

from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths


class OldDependencies(sp.ResourceSelector):
    name = 'old_dependencies_resource_id'
    description = 'Old dependencies'
    resource_type = resource_types.DIR_LIST
    group = 'Task resources'


class NewDependencies(sp.ResourceSelector):
    name = 'new_dependencies_resource_id'
    description = 'New dependencies'
    resource_type = resource_types.DIR_LIST
    group = 'Task resources'


class CompareDependencies(SandboxTask):
    """
        Сравнивает два списка зависимостей.
    """
    type = 'COMPARE_DEPENDENCIES'

    execution_space = 256  # 256 Mb
    ram = 128
    cores = 1

    input_parameters = [
        OldDependencies,
        NewDependencies,
    ]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self.create_resource(
            description="Deps diff for: " + self.descr,
            resource_path='dependencies.diff',
            resource_type=resource_types.UNIFIED_GNU_DIFF_FILE,
            attributes={
                'ttl': 40,  # they are small, let them live longer
                'backup_task': True,  # they are important, so auto-backup them
            },
        )
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        old_output = self.sync_resource(self.ctx[OldDependencies.name])
        new_output = self.sync_resource(self.ctx[NewDependencies.name])

        diff_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])
        diff_handle = open(diff_resource.path, 'w')
        diff_err_handle = open(os.path.join(paths.get_logs_folder(), 'diff.err'), 'w')
        diff_proc = process.run_process(
            [
                'diff',
                '-u',
                old_output,
                new_output,
            ],
            stdout=diff_handle,
            stderr=diff_err_handle,
            outputs_to_one_file=False,
            check=False,
        )
        diff_handle.close()
        diff_err_handle.close()

        self.ctx['changed_dependencies'] = diff_proc.returncode != 0

    def get_short_task_result(self):
        if not self.is_completed():
            return None
        return "diff" if self.ctx.get('changed_dependencies', False) else "no diff"


__Task__ = CompareDependencies
